package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.group.ReleaseGroup
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseGroupSchema.{COLUMN_LENGTH_TITLE, RELEASE_GROUPS}
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseGroupSqlBuilder.STMT_RELEASE_GROUP_SELECT
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichStringAsTruncatable, params}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class ReleaseGroupPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                           @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                           folderPersistence: FolderPersistence)
  extends PersistenceSupport {

  private val STMT_GET_UID_BY_ID = s"SELECT ${RELEASE_GROUPS.CI_UID} FROM ${RELEASE_GROUPS.TABLE} WHERE ${RELEASE_GROUPS.ID} = :id"

  def getUid(groupId: CiId): Option[CiUid] =
    sqlQuery(STMT_GET_UID_BY_ID, params("id" -> getName(groupId.normalized)), rs => CiUid(rs.getInt(RELEASE_GROUPS.CI_UID))).headOption

  private val STMT_INSERT_RELEASE_GROUP =
    s"""|INSERT INTO ${RELEASE_GROUPS.TABLE}
        |   ( ${RELEASE_GROUPS.ID}
        |   , ${RELEASE_GROUPS.FOLDER_UID}
        |   , ${RELEASE_GROUPS.TITLE}
        |   , ${RELEASE_GROUPS.STATUS}
        |   , ${RELEASE_GROUPS.START_DATE}
        |   , ${RELEASE_GROUPS.END_DATE}
        |   , ${RELEASE_GROUPS.RISK_SCORE}
        |   , ${RELEASE_GROUPS.CONTENT}
        |   )
        | VALUES
        |   ( :id
        |   , :folderUid
        |   , :title
        |   , :status
        |   , :startDate
        |   , :endDate
        |   , :riskScore
        |   , :content
        |   )
        """.stripMargin

  def insert(releaseGroup: ReleaseGroup): CiUid = {
    val folderUid = folderPersistence.getUid(releaseGroup.getFolderId)
    val content = serialize(releaseGroup)
    try {
      sqlInsertWithContent(pkName(RELEASE_GROUPS.CI_UID), STMT_INSERT_RELEASE_GROUP, params(
        "id" -> getName(releaseGroup.getId.normalized),
        "folderUid" -> folderUid,
        "title" -> releaseGroup.getTitle.truncate(COLUMN_LENGTH_TITLE),
        "status" -> releaseGroup.getStatus.value(),
        "startDate" -> releaseGroup.getStartDate,
        "endDate" -> releaseGroup.getEndDate,
        "riskScore" -> releaseGroup.getRiskScore
      ), "content" -> content,
        ciUid => ciUid
      )
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Release group with ID '${releaseGroup.getId}' already exists", ex)
    }
  }

  private val STMT_UPDATE_RELEASE_GROUP =
    s"""|UPDATE ${RELEASE_GROUPS.TABLE}
        | SET
        |  ${RELEASE_GROUPS.TITLE} = :newTitle,
        |  ${RELEASE_GROUPS.FOLDER_UID} = :folderUid,
        |  ${RELEASE_GROUPS.STATUS} = :newStatus,
        |  ${RELEASE_GROUPS.START_DATE} = :newStartDate,
        |  ${RELEASE_GROUPS.END_DATE} = :newEndDate,
        |  ${RELEASE_GROUPS.RISK_SCORE} = :newRiskScore,
        |  ${RELEASE_GROUPS.CONTENT} = :content
        | WHERE
        |  ${RELEASE_GROUPS.ID} = :id
        """.stripMargin

  def update(releaseGroup: ReleaseGroup): Unit = {
    val folderUid = folderPersistence.getUid(releaseGroup.getFolderId)
    val content = serialize(releaseGroup)
    sqlExecWithContent(STMT_UPDATE_RELEASE_GROUP, params(
      "id" -> getName(releaseGroup.getId.normalized),
      "folderUid" -> folderUid,
      "newTitle" -> releaseGroup.getTitle.truncate(COLUMN_LENGTH_TITLE),
      "newStatus" -> releaseGroup.getStatus.value(),
      "newStartDate" -> releaseGroup.getStartDate,
      "newEndDate" -> releaseGroup.getEndDate,
      "newRiskScore" -> releaseGroup.getRiskScore
    ), "content" -> content,
      checkCiUpdated(releaseGroup.getId)
    )
  }

  private val STMT_DELETE_RELEASE_GROUP_BY_ID =
    s"""|DELETE FROM ${RELEASE_GROUPS.TABLE}
        | WHERE ${RELEASE_GROUPS.ID} = :id""".stripMargin

  def delete(groupId: CiId): Unit = sqlExec(STMT_DELETE_RELEASE_GROUP_BY_ID, params("id" -> getName(groupId.normalized)), ps => Try(ps.execute()))

  private val STMT_EXISTS_RELEASE_GROUP_BY_ID: String = s"SELECT COUNT(*) FROM ${RELEASE_GROUPS.TABLE} WHERE ${RELEASE_GROUPS.ID} = :id"

  def exists(groupId: CiId): Boolean = sqlQuery(STMT_EXISTS_RELEASE_GROUP_BY_ID, params("id" -> getName(groupId.normalized)), _.getInt(1) > 0).head

  private val STMT_FIND_RELEASE_GROUP_BY_ID =
    s"""|$STMT_RELEASE_GROUP_SELECT
        | WHERE rg.${RELEASE_GROUPS.ID} = ?""".stripMargin

  def findById(id: CiId): Option[ReleaseGroupRow] = {
    findOptional(_.queryForObject(STMT_FIND_RELEASE_GROUP_BY_ID, releaseGroupRowMapper, getName(id.normalized)))
  }

  def findByQuery(sqlWithParameters: SqlWithParameters): Seq[ReleaseGroupRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[ReleaseGroupRow](sql, params.toArray, releaseGroupRowMapper).asScala.toSeq
  }

  private val STMT_GET_FOLDER_ID_BY_GROUP_ID =
    s"""| SELECT f.${FOLDERS.FOLDER_ID}
        | FROM ${RELEASE_GROUPS.TABLE} rg
        | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = rg.${RELEASE_GROUPS.FOLDER_UID}
        | WHERE rg.${RELEASE_GROUPS.ID} = :groupId""".stripMargin

  def findFolderId(groupId: String): FolderId = {
    sqlQuery(STMT_GET_FOLDER_ID_BY_GROUP_ID, params("groupId" -> getName(groupId.normalized)),
      rs => Option(rs.getString(1)).map(FolderId(_)).getOrElse(FolderId.Root)
    ).headOption.getOrElse(throw new LogFriendlyNotFoundException(s"Folder of group $groupId not found"))
  }

  private val releaseGroupRowMapper: RowMapper[ReleaseGroupRow] = (rs: ResultSet, _: Int) => {
    val contentStream = rs.getBinaryStream(RELEASE_GROUPS.CONTENT)
    try {
      val content = decompress(contentStream)
      ReleaseGroupRow(
        content,
        ciUid = rs.getInt(RELEASE_GROUPS.CI_UID),
        folderId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID)
      )
    } finally {
      contentStream.close()
    }
  }

  private def serialize(releaseGroup: ReleaseGroup): String = {
    val releaseGroupCopy = cloneCi(releaseGroup)
    releaseGroupCopy.setFolderId(null)
    val content = CiSerializerHelper.serialize(releaseGroupCopy)
    content
  }
}
