package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.group.ReleaseGroup.GROUP_ROOT
import com.xebialabs.xlrelease.domain.group.ReleaseGroupStatus
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.ReleaseGroupSchema.{RELEASE_GROUPS, RELEASE_GROUP_MEMBERS}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class ReleaseGroupMemberPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                 @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect,
                                                 releaseGroupPersistence: ReleaseGroupPersistence,
                                                 folderPersistence: FolderPersistence) extends PersistenceSupport {

  private val STMT_FIND_GROUPS_REFERENCING_MEMBER_ID: String =
    s"""SELECT DISTINCT rg.${RELEASE_GROUPS.ID}
       |FROM ${RELEASE_GROUP_MEMBERS.TABLE} rgm
       |INNER JOIN ${RELEASE_GROUPS.TABLE} rg ON rg.${RELEASE_GROUPS.CI_UID} = rgm.${RELEASE_GROUP_MEMBERS.GROUP_UID}
       |WHERE rgm.${RELEASE_GROUP_MEMBERS.RELEASE_ID} = :releaseId""".stripMargin

  def findGroupsReferencingReleaseId(releaseId: CiId): Seq[CiId] = {
    sqlQuery(STMT_FIND_GROUPS_REFERENCING_MEMBER_ID, params("releaseId" -> getName(releaseId.normalized)), _.getString(1)).map(name => s"$GROUP_ROOT/$name").toSeq
  }

  private val STMT_FIND_DESCENDANTS_IDS_BY_FOLDER_ID =
    s"""SELECT fp.DESCENDANT_UID
       |FROM XLR_FOLDERS f
       |INNER JOIN XLR_FOLDER_PATHS fp ON f.CI_UID = fp.ANCESTOR_UID
       |WHERE f.FOLDER_ID = (:folderId)""".stripMargin

  private val STMT_FIND_ACTIVE_GROUPS_BY_FOLDER_ID: String =
    s"""SELECT DISTINCT rg.${RELEASE_GROUPS.ID}, rg.${RELEASE_GROUPS.TITLE}
       |FROM ${RELEASE_GROUP_MEMBERS.TABLE} rgm
       |INNER JOIN ${RELEASE_GROUPS.TABLE} rg ON rg.${RELEASE_GROUPS.CI_UID} = rgm.${RELEASE_GROUP_MEMBERS.GROUP_UID}
       |INNER JOIN ${RELEASES.TABLE} r ON r.${RELEASES.RELEASE_ID} = rgm.${RELEASE_GROUP_MEMBERS.RELEASE_ID}
       |INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |WHERE f.${FOLDERS.CI_UID} IN (SELECT DESCENDANT_UID from ($STMT_FIND_DESCENDANTS_IDS_BY_FOLDER_ID) fd)
       |AND rg.${RELEASE_GROUPS.FOLDER_UID} NOT IN (SELECT DESCENDANT_UID from ($STMT_FIND_DESCENDANTS_IDS_BY_FOLDER_ID) fd)
       |AND rg.${RELEASE_GROUPS.STATUS} NOT IN (:inactiveStatuses)""".stripMargin

  def findActiveGroupsReferencingFolderId(folderId: CiId): Seq[CiIdWithTitle] = {
    sqlQuery(STMT_FIND_ACTIVE_GROUPS_BY_FOLDER_ID,
      params("folderId" -> FolderId(folderId).id,
        "depth" -> Int.MaxValue,
        "inactiveStatuses" -> ReleaseGroupStatus.INACTIVE_STATUSES.map(_.value()).toSeq.asJava
      ), rs => CiIdWithTitle(s"$GROUP_ROOT/${rs.getString(1)}", rs.getString(2))).toSeq
  }

  private val STMT_FIND_MEMBERS_BY_GROUP_UIDS =
    s"""SELECT rgm.${RELEASE_GROUP_MEMBERS.GROUP_UID}, f.${FOLDERS.FOLDER_PATH}, f.${FOLDERS.FOLDER_ID}, rgm.${RELEASE_GROUP_MEMBERS.RELEASE_ID}
       |FROM ${RELEASE_GROUP_MEMBERS.TABLE} rgm
       |INNER JOIN ${RELEASES.TABLE} r ON r.${RELEASES.RELEASE_ID} = rgm.${RELEASE_GROUP_MEMBERS.RELEASE_ID}
       |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |WHERE rgm.${RELEASE_GROUP_MEMBERS.GROUP_UID} IN (:groupUids)""".stripMargin

  def findMembersByGroupUid(groupUid: CiUid): Seq[String] = findMembersByGroupUids(Seq(groupUid)).getOrElse(groupUid, Seq.empty)

  def findMembersByGroupUids(groupUids: Seq[CiUid]): Map[CiUid, Seq[String]] = if (groupUids.nonEmpty) {
    sqlQuery(STMT_FIND_MEMBERS_BY_GROUP_UIDS, params("groupUids" -> groupUids.asJava), rs => {
      val groupUid = CiUid(rs.getString(RELEASE_GROUP_MEMBERS.GROUP_UID))
      val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
      val releaseId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / rs.getString(RELEASE_GROUP_MEMBERS.RELEASE_ID)
      (groupUid, releaseId.absolute)
    }).groupBy(_._1).view.mapValues(_.map(_._2).toSeq).toMap
  } else {
    Map.empty
  }

  private val STMT_INSERT_MEMBERS: String =
    s"""
       |INSERT INTO ${RELEASE_GROUP_MEMBERS.TABLE} (${RELEASE_GROUP_MEMBERS.GROUP_UID}, ${RELEASE_GROUP_MEMBERS.RELEASE_ID})
       |SELECT :groupUid, r.${RELEASES.RELEASE_ID}
       |FROM ${RELEASES.TABLE} r
       |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin // on purpose like this so we do not re-insert archived releases

  def insertMembers(groupUid: CiUid, releaseIds: Set[CiId]): Unit = if (releaseIds.nonEmpty) {
    sqlBatch(STMT_INSERT_MEMBERS, releaseIds.map { id =>
      Map("groupUid" -> groupUid, "releaseId" -> getName(id.normalized))
    })
  }

  private val STMT_DELETE_GROUP_MEMBERS: String =
    s"""
       |DELETE FROM ${RELEASE_GROUP_MEMBERS.TABLE}
       | WHERE
       |  ${RELEASE_GROUP_MEMBERS.GROUP_UID} = :groupUid
       |  AND ${RELEASE_GROUP_MEMBERS.RELEASE_ID} = :releaseId
       |""".stripMargin

  def deleteMembers(groupUid: CiUid, releaseIds: Set[CiId]): Unit = if (releaseIds.nonEmpty) {
    sqlBatch(STMT_DELETE_GROUP_MEMBERS, releaseIds.map(id => Map("groupUid" -> groupUid, "releaseId" -> getName(id.normalized))))
  }

  private val STMT_DELETE_GROUP_MEMBERS_BY_RELEASE_ID: String =
    s"""
       |DELETE FROM ${RELEASE_GROUP_MEMBERS.TABLE}
       | WHERE
       |  ${RELEASE_GROUP_MEMBERS.RELEASE_ID} = :releaseId
       |""".stripMargin

  def deleteMembersByReleaseId(releaseId: CiId): Unit =
    sqlExec(STMT_DELETE_GROUP_MEMBERS_BY_RELEASE_ID, Map("releaseId" -> getName(releaseId.normalized)), ps => Try(ps.execute()))

}
