package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseGroupFilters, ReleaseGroupOrderMode}
import com.xebialabs.xlrelease.api.v1.views.ReleaseGroupTimeline
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.domain.group.{ReleaseGroup, ReleaseGroupStatus}
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.ReleaseStatus._
import com.xebialabs.xlrelease.events.{ReleaseGroupCreatedEvent, ReleaseGroupDeletedEvent, ReleaseGroupUpdatedEvent, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.Ids.{isDomainId, isReleaseId, isRoot}
import com.xebialabs.xlrelease.repository.sql.persistence.CiIdWithTitle
import com.xebialabs.xlrelease.repository.{Page, ReleaseGroupRepository, ReleaseRepository}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class ReleaseGroupService @Autowired()(releaseGroupRepository: ReleaseGroupRepository,
                                       val releaseRepository: ReleaseRepository,
                                       val releaseService: ReleaseService,
                                       ciIdService: CiIdService,
                                       archivedReleases: ArchivedReleases,
                                       folderService: FolderService,
                                       roleService: RoleService,
                                       val eventBus: XLReleaseEventBus) extends Logging with ReleaseGroupTimelineCalculator {
  @Timed
  def getGroup(groupId: String): ReleaseGroup = releaseGroupRepository.read(groupId)

  @Timed
  def existsGroup(groupId: String): Boolean = releaseGroupRepository.exists(groupId)

  @Timed
  def createGroup(releaseGroup: ReleaseGroup): ReleaseGroup = {
    logger.debug(s"Creating new release group [$releaseGroup]")
    validate(releaseGroup)
    releaseGroup.setId(ciIdService.getUniqueId(Type.valueOf(classOf[ReleaseGroup]), ReleaseGroup.GROUP_ROOT))
    recalculateStatusAndProgress(releaseGroup)
    recalculateRiskScore(releaseGroup)
    releaseGroupRepository.create(releaseGroup)
    eventBus.publish(ReleaseGroupCreatedEvent(releaseGroup))
    releaseGroup
  }

  @Timed
  def updateGroup(releaseGroup: ReleaseGroup): ReleaseGroup = {
    logger.debug(s"Updating release group [$releaseGroup]")
    validate(releaseGroup)
    checkIsUpdatable(releaseGroupRepository.read(releaseGroup.getId))

    recalculateStatusAndProgress(releaseGroup)
    recalculateRiskScore(releaseGroup)
    releaseGroupRepository.update(releaseGroup)
    eventBus.publish(ReleaseGroupUpdatedEvent(releaseGroup))
    releaseGroup
  }

  @Timed
  def deleteGroup(groupId: String): Unit = {
    logger.debug(s"Deleting release group [$groupId]")
    if (releaseGroupRepository.exists(groupId)) {
      val releaseGroup = releaseGroupRepository.read(groupId)
      checkIsUpdatable(releaseGroup, "delete")
      releaseGroupRepository.delete(groupId)
      eventBus.publish(ReleaseGroupDeletedEvent(releaseGroup))
    } else {
      logger.debug(s"Release group [$groupId] already deleted")
    }
  }

  @Timed
  def addMembersToGroup(groupId: String, memberIds: Seq[String]): Unit = {
    logger.debug(s"Adding members $memberIds to release group [$groupId]")
    memberIds.foreach(validateMember)
    val releaseGroup = releaseGroupRepository.read(groupId)
    checkIsUpdatable(releaseGroup)

    releaseGroup.getReleaseIds.addAll(memberIds.asJava)
    recalculateStatusAndProgress(releaseGroup)
    recalculateRiskScore(releaseGroup)
    releaseGroupRepository.update(releaseGroup)
    eventBus.publish(ReleaseGroupUpdatedEvent(releaseGroup))
  }

  @Timed
  def removeMembersFromGroup(groupId: String, memberIds: Seq[String]): Unit = {
    logger.debug(s"Removing members $memberIds from release group [$groupId]")
    val releaseGroup = releaseGroupRepository.read(groupId)
    checkIsUpdatable(releaseGroup)

    releaseGroup.removeReleaseIds(memberIds.asJava)
    recalculateStatusAndProgress(releaseGroup)
    recalculateRiskScore(releaseGroup)
    releaseGroupRepository.update(releaseGroup)
    eventBus.publish(ReleaseGroupUpdatedEvent(releaseGroup))
  }

  @Timed
  def search(filters: ReleaseGroupFilters, page: Page, orderBy: ReleaseGroupOrderMode): java.util.List[ReleaseGroup] =
    releaseGroupRepository.search(filters, page, orderBy, currentPrincipals, currentRoleIds).asJava

  @Timed
  def getTimeline(groupId: String, now: DateTime): ReleaseGroupTimeline = {
    val releaseGroup = getGroup(groupId)
    calculateTimeline(releaseGroup, now)
  }

  @Timed
  def updateGroupStatus(groupId: String): Unit = {
    logger.debug(s"Recalculating status for release group [$groupId]")
    // TODO: use lighter read and update methods
    val releaseGroup = releaseGroupRepository.read(groupId)
    recalculateStatusAndProgress(releaseGroup)
    releaseGroupRepository.update(releaseGroup)
  }

  @Timed
  def updateGroupRisk(groupId: String): Unit = {
    logger.debug(s"Recalculating risk for release group [$groupId]")
    // TODO: use lighter read and update methods
    val releaseGroup = releaseGroupRepository.read(groupId)
    recalculateRiskScore(releaseGroup)
    releaseGroupRepository.update(releaseGroup)
  }

  private def recalculateRiskScore(releaseGroup: ReleaseGroup): Unit = {
    val releaseIds = releaseGroup.getReleaseIds.asScala.toSeq
    val riskScores = getReleaseRiskScores(releaseIds)
    releaseGroup.setRiskScore(Try(riskScores.sum / riskScores.length).getOrElse[Int](0))
    logger.debug(s"Release group [${releaseGroup.getId}] risk score is now: ${releaseGroup.getRiskScore}")
  }

  def getReleaseRiskScores(releaseIds: Seq[String]): Seq[Int] = {
    if (releaseIds.nonEmpty) releaseRepository.getRiskScores(releaseIds).padTo(releaseIds.size, 0) else Seq.empty
  }

  def getFolderId(groupId: String): String = releaseGroupRepository.findFolderId(groupId)

  @Timed
  def findGroupsReferencingRelease(releaseId: String): Seq[String] = releaseGroupRepository.findGroupsReferencingReleaseId(releaseId)

  @Timed
  def findActiveGroupsReferencingFolder(folderId: String): Seq[CiIdWithTitle] = releaseGroupRepository.findActiveGroupsReferencingFolderId(folderId)


  private def recalculateStatusAndProgress(releaseGroup: ReleaseGroup): Unit = {
    val releaseIds = releaseGroup.getReleaseIds.asScala.toSeq
    val statuses = getReleaseStatuses(releaseIds)

    releaseGroup.setStatus(calculateGroupStatus(releaseGroup, statuses))
    logger.debug(s"Release group [${releaseGroup.getId}] status is now: ${releaseGroup.getStatus}")
    releaseGroup.setProgress(calculateGroupProgress(statuses))
    logger.debug(s"Release group [${releaseGroup.getId}] progress is now: ${releaseGroup.getProgress}")
  }

  private def getReleaseStatuses(releaseIds: Seq[String]): Seq[ReleaseStatus] = {
    if (releaseIds.nonEmpty) releaseRepository.getStatuses(releaseIds).padTo(releaseIds.size, ReleaseStatus.COMPLETED) else Seq.empty
  }

  private def calculateGroupProgress(statuses: Seq[ReleaseStatus]): Integer = {
    if (statuses.isEmpty) 0 else ((statuses.count(_.isInactive) / statuses.size.toDouble) * 100).round.toInt
  }

  private def calculateGroupStatus(releaseGroup: ReleaseGroup, statuses: Seq[ReleaseStatus]): ReleaseGroupStatus = {
    if (statuses.isEmpty) {
      ReleaseGroupStatus.PLANNED
    } else if (statuses.forall(_ == statuses.head)) {
      // all are STATUS -> STATUS
      ReleaseGroupStatus.fromRelease(statuses.head)
    } else if (statuses.contains(FAILING) || statuses.contains(FAILED)) {
      // at least one is FAILING, FAILED -> FAILING
      ReleaseGroupStatus.FAILING
    } else if (statuses.contains(IN_PROGRESS) || statuses.contains(PLANNED) || statuses.contains(PAUSED)) {
      // at least one is IN_PROGRESS|PLANNED|PAUSED and no FAILING, FAILED and not all are same -> IN_PROGRESS
      ReleaseGroupStatus.IN_PROGRESS
    } else if (!statuses.exists(status => status != COMPLETED && status != ABORTED)) {
      // all are either COMPLETED or ABORTED -> COMPLETED
      ReleaseGroupStatus.COMPLETED
    } else {
      // catch all
      releaseGroup.getStatus
    }
  }

  private def checkIsUpdatable(existingGroup: ReleaseGroup, action: String = "update"): Unit = {
    checkArgument(existingGroup.isUpdatable, s"Cannot $action release group '${existingGroup.getTitle}' because it is ${existingGroup.getStatus}")
  }

  private def validate(releaseGroup: ReleaseGroup): Unit = {
    checkNotNull(releaseGroup, "Release group")
    checkArgument(releaseGroup.getStartDate != null, "Start date must be set")
    checkArgument(releaseGroup.getEndDate != null, "End date must be set")
    checkArgument(isNotBlank(releaseGroup.getTitle), "Title must be set")
    checkArgument(releaseGroup.getTitle.length < 256, "Title must be 255 characters or less")
    checkArgument(releaseGroup.getEndDate.after(releaseGroup.getStartDate), "End date must be after start date")
    checkArgument(isNotBlank(releaseGroup.getFolderId), "Folder ID must be set")
    checkArgument(!isRoot(releaseGroup.getFolderId), s"Provided folder ID '${releaseGroup.getFolderId}' must not be a root folder")
    if (releaseGroup.getReleaseIds == null) {
      releaseGroup.setReleaseIds(new java.util.HashSet[String])
    }
    releaseGroup.getReleaseIds.forEach(validateMember)
    checkArgument(folderService.exists(releaseGroup.getFolderId), s"Provided folder ID '${releaseGroup.getFolderId}' must exist in the database")
  }

  private def validateMember(releaseId: String): Unit = {
    checkArgument(isDomainId(releaseId) && isReleaseId(releaseId), s"Provided ID '$releaseId' must be a valid release ID")
    checkArgument(releaseRepository.exists(releaseId) || archivedReleases.exists(releaseId), s"Provided ID '$releaseId' must exist in the database")
  }

  private def currentPrincipals = authenticationToPrincipals(getAuthentication).asScala

  private def currentRoleIds = roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
}
