package com.xebialabs.xlrelease.events.handler

import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.xlrelease.actors.ReleaseGroupActorService
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, SynchronizedSubscribe}
import com.xebialabs.xlrelease.risk.domain.events.RiskScoreUpdated
import com.xebialabs.xlrelease.service.ReleaseGroupService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component


@Component
@EventListener
class ReleaseGroupEventHandler @Autowired()(releaseGroupActorService: ReleaseGroupActorService,
                                            releaseGroupService: ReleaseGroupService) {
  @SynchronizedSubscribe
  def onReleaseExecution(event: ReleaseExecutionEvent): Unit = triggerStatusCalculation(event.release)

  @SynchronizedSubscribe
  def onRiskScoreUpdated(event: RiskScoreUpdated): Unit = {
    releaseGroupService.findGroupsReferencingRelease(event.releaseId).foreach(releaseGroupActorService.updateGroupRisk)
  }

  @SynchronizedSubscribe
  def onFolderDeleting(action: FolderDeletingAction): Unit = {
    val groups = releaseGroupService.findActiveGroupsReferencingFolder(action.folderId)
    if (groups.nonEmpty) {
      throw new ItemInUseException(s"You cannot delete this folder. The folder or its subfolders contain releases referenced by active groups: " +
        s"${groups.take(20).map(_.title).sorted.mkString("\"", "\", \"", "\"")}.")
    }
  }

  private def triggerStatusCalculation(release: Release): Unit = {
    releaseGroupService.findGroupsReferencingRelease(release.getId).foreach(releaseGroupActorService.updateGroupStatus)
  }
}
