package com.xebialabs.xlrelease.actors

import akka.actor.{Actor, ActorLogging, Props}
import com.xebialabs.xlrelease.actors.ReleaseGroupMessages._
import com.xebialabs.xlrelease.domain.group.ReleaseGroup
import com.xebialabs.xlrelease.service.ReleaseGroupService
import grizzled.slf4j.Logging

import scala.util.{Failure, Success, Try}

object ReleaseGroupMessages {

  case class CreateGroup(group: ReleaseGroup)

  case class UpdateGroup(group: ReleaseGroup)

  case class DeleteGroup(groupId: String)

  case class AddMembersToGroup(groupId: String, memberIds: Seq[String])

  case class RemoveMembersFromGroup(groupId: String, memberIds: Seq[String])

  case class UpdateGroupStatus(groupId: String)

  case class UpdateGroupRisk(groupId: String)

}

object ReleaseGroupProcessingActor {
  def props(releaseGroupService: ReleaseGroupService) = Props(new ReleaseGroupProcessingActor(releaseGroupService)).withDispatcher("xl.dispatchers.release-dispatcher")

  def name = "release-group-processing-actor"
}

class ReleaseGroupProcessingActor(releaseGroupService: ReleaseGroupService)
  extends Actor with ActorLogging with Logging with ExceptionTranslateActor {

  override def receive: Actor.Receive = {
    case CreateGroup(group) => replyOrFail {
      releaseGroupService.createGroup(group)
    }
    case UpdateGroup(group) => replyOrFail {
      releaseGroupService.updateGroup(group)
    }
    case DeleteGroup(groupId) => replyOrFail {
      releaseGroupService.deleteGroup(groupId)
    }
    case AddMembersToGroup(groupId, memberIds) => replyOrFail {
      releaseGroupService.addMembersToGroup(groupId, memberIds)
    }
    case RemoveMembersFromGroup(groupId, memberIds) => replyOrFail {
      releaseGroupService.removeMembersFromGroup(groupId, memberIds)
    }
    case UpdateGroupStatus(groupId) => replyOrFail {
      releaseGroupService.updateGroupStatus(groupId)
    }
    case UpdateGroupRisk(groupId) => replyOrFail {
      releaseGroupService.updateGroupRisk(groupId)
    }
  }

  private def replyOrFail[T](call: => T): Unit = sender() ! (Try(call) match {
    case Success(t) if t != null => t
    case Success(_) => akka.actor.Status.Failure(new NullPointerException("Method returned null and this cannot be processed"))
    case Failure(ex) =>
      logger.warn("Failed to process release group message", ex)
      akka.actor.Status.Failure(translate(ex))
  })

}
