package com.xebialabs.xlrelease.api.v1.impl;

import java.util.ArrayList;
import java.util.List;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlrelease.actors.ReleaseGroupActorService;
import com.xebialabs.xlrelease.api.v1.ReleaseGroupApi;
import com.xebialabs.xlrelease.api.v1.forms.ReleaseGroupFilters;
import com.xebialabs.xlrelease.api.v1.forms.ReleaseGroupOrderMode;
import com.xebialabs.xlrelease.api.v1.views.ReleaseGroupTimeline;
import com.xebialabs.xlrelease.domain.group.ReleaseGroup;
import com.xebialabs.xlrelease.repository.Page;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ReleaseGroupService;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_GROUP;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.VIEW_RELEASE_GROUP;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

@Controller
public class ReleaseGroupApiImpl implements ReleaseGroupApi {
    private final ReleaseGroupActorService releaseGroupActorService;
    private final ReleaseGroupService releaseGroupService;
    private final PermissionChecker permissions;

    @Autowired
    public ReleaseGroupApiImpl(ReleaseGroupActorService releaseGroupActorService, ReleaseGroupService releaseGroupService, PermissionChecker permissions) {
        this.releaseGroupActorService = releaseGroupActorService;
        this.releaseGroupService = releaseGroupService;
        this.permissions = permissions;
    }

    @Timed
    @Override
    public ReleaseGroup getGroup(String groupId) {
        checkPermissions(groupId, VIEW_RELEASE_GROUP);
        return releaseGroupService.getGroup(groupId);
    }

    @Timed
    @Override
    public void deleteGroup(String groupId) {
        checkPermissions(groupId, EDIT_RELEASE_GROUP);
        // TODO is there a state in which we cannot delete the group anymore?
        releaseGroupActorService.deleteGroup(groupId);
    }

    @Timed
    @Override
    public ReleaseGroup createGroup(ReleaseGroup releaseGroup) {
        checkNotNull(releaseGroup.getFolderId(), "folderId");
        permissions.check(EDIT_RELEASE_GROUP, releaseGroup.getFolderId());
        return releaseGroupActorService.createGroup(releaseGroup);
    }

    @Timed
    @Override
    public ReleaseGroup updateGroup(String groupId, ReleaseGroup releaseGroup) {
        releaseGroup.setId(groupId);
        checkPermissions(groupId, EDIT_RELEASE_GROUP);
        return releaseGroupActorService.updateGroup(releaseGroup);
    }

    // TODO: api/v1/release-groups/ReleaseGroup123/members or api/v1/release-groups/ReleaseGroup123/releases, ReleaseGroup.memberIds or ReleaseGroup.releaseIds
    @Timed
    @Override
    public List<String> getMembers(String groupId) {
        checkNotNull(groupId, "groupId");
        checkPermissions(groupId, VIEW_RELEASE_GROUP);

        return new ArrayList<>(releaseGroupService.getGroup(groupId).getReleaseIds());
    }

    @Timed
    @Override
    public void addMembersToGroup(String groupId, List<String> memberIds) {
        checkNotNull(groupId, "groupId");
        checkNotNull(memberIds, "memberIds");
        checkPermissions(groupId, EDIT_RELEASE_GROUP);

        releaseGroupActorService.addMembersToGroup(groupId, memberIds);
    }

    @Timed
    @Override
    public void removeMembersFromGroup(String groupId, List<String> memberIds) {
        checkNotNull(groupId, "groupId");
        checkNotNull(memberIds, "memberIds");
        checkPermissions(groupId, EDIT_RELEASE_GROUP);

        releaseGroupActorService.removeMembersFromGroup(groupId, memberIds);
    }

    @Timed
    @Override
    public List<ReleaseGroup> searchGroups(ReleaseGroupFilters groupFilters,
                                           Long page,
                                           Long resultsPerPage,
                                           ReleaseGroupOrderMode orderBy) {
        checkArgument(resultsPerPage <= 100, "Number of results per page cannot be more than 100");

        if (null == groupFilters) {
            groupFilters = new ReleaseGroupFilters();
        }
        return releaseGroupService.search(groupFilters, Page.parse(ofNullable(page), ofNullable(resultsPerPage), empty()), orderBy);
    }

    @Timed
    @Override
    public ReleaseGroupTimeline getReleaseGroupTimeline(String groupId) {
        checkPermissions(groupId, VIEW_RELEASE_GROUP);
        return releaseGroupService.getTimeline(groupId, DateTime.now());
    }

    private void checkPermissions(String groupId, Permission permission) {
        String folderId = releaseGroupService.getFolderId(groupId);
        permissions.check(permission, folderId);
    }
}
