import React from 'react';
import { ReactWrapper } from 'enzyme';
import { stringPropertyMock } from '@xlr-ui/app/features/tasks/__mocks__/property.mock';
import { taskMock } from '@xlr-ui/app/features/tasks/__mocks__/task.mock';
import { mountWithStoreAndTheme } from '@xlr-ui/tests/unit/testing-utils';
import { FolderWithVariables } from './folder.component';
import { StringWithVariables } from './string-with-variables.component';
import { InputWithFolder } from '@xlr-ui/app/features/tasks/components/rails/config/components/input-with-folder/input-with-folder.component';
import { PropertiesEnum } from '@xlr-ui/app/features/tasks/components/rails/config/components/types';

describe('Folder component', () => {
    const state = {
        taskDrawer: {},
        folders: {
            folders: [
                { id: 'Folder1', title: 'Folder One' },
                { id: 'Folder2', title: 'Folder Two' },
            ],
        },
    };
    const task = {
        ...taskMock,
        inputProperties: {
            string: 'Folder2',
        },
    };
    const onChange = jest.fn();
    const dispatch = jest.fn();
    let wrapper: ReactWrapper;

    const getStringVariables = () => wrapper.find(StringWithVariables);
    const getInputWithFolder = () => wrapper.find(InputWithFolder);

    beforeEach(() => {
        wrapper = mountWithStoreAndTheme(
            <FolderWithVariables
                disabled={false}
                expanded={false}
                onChange={onChange}
                propertiesEnum={PropertiesEnum.InputProperties}
                property={stringPropertyMock}
                task={task}
            />,
            dispatch,
            state,
        );
    });

    it('should allow variables', () => {
        const stringVariablesComponent = getStringVariables();
        const stringVariablesProps = stringVariablesComponent.props();
        expect(stringVariablesComponent).toExist();
        expect(stringVariablesProps.disabled).toBe(false);
        expect(stringVariablesProps.expanded).toBe(false);
        expect(stringVariablesProps.onChange).toBe(onChange);
        expect(stringVariablesProps.property).toBe(stringPropertyMock);
        expect(stringVariablesProps.task).toBe(task);
    });

    it('should set the folder selected', () => {
        expect(getInputWithFolder().props().value).toMatchObject({ id: 'Folder2', title: 'Folder Two' });
    });

    it('should set error if the folder selected can not be found', () => {
        const taskWithUnknownFolder = {
            ...taskMock,
            inputProperties: {
                string: 'FolderXXXXX',
            },
        };
        wrapper = mountWithStoreAndTheme(
            <FolderWithVariables
                disabled={false}
                expanded={false}
                onChange={onChange}
                propertiesEnum={PropertiesEnum.InputProperties}
                property={stringPropertyMock}
                task={taskWithUnknownFolder}
            />,
            dispatch,
            state,
        );
        const props = getInputWithFolder().props();
        expect(props.value).toMatchObject({ id: 'FolderXXXXX', title: 'FolderXXXXX' });
        expect(props.error).toBe(true);
        expect(props.description).toBe("You do not have permissions to view the selected folder or it doesn't exist");
    });

    it('should set the props to the InputWithFolder', () => {
        const inputWithFolderProps = getInputWithFolder().props();
        expect(inputWithFolderProps.description).toBe(stringPropertyMock.description);
        expect(inputWithFolderProps.disabled).toBe(false);
        expect(inputWithFolderProps.isFolderFullWidth).toBe(false);
        expect(inputWithFolderProps.label).toBe(stringPropertyMock.label);
        expect(inputWithFolderProps.required).toBe(stringPropertyMock.required);
    });

    it('should handle onChange with folder', () => {
        const inputWithFolderProps = getInputWithFolder().props();
        inputWithFolderProps.onChange?.({ id: 'Folder1', title: 'Folder One' });
        expect(onChange).toHaveBeenCalledWith('Folder1');
    });

    it('should handle onChange with undefined', () => {
        const inputWithFolderProps = getInputWithFolder().props();
        inputWithFolderProps.onChange?.(undefined);
        expect(onChange).toHaveBeenCalledWith(null);
    });

    it('should not have value if task does not have input properties', () => {
        const testTask = { ...task, inputProperties: undefined };
        wrapper = mountWithStoreAndTheme(
            <FolderWithVariables
                disabled={false}
                expanded={false}
                onChange={onChange}
                propertiesEnum={PropertiesEnum.InputProperties}
                property={stringPropertyMock}
                task={testTask}
            />,
            dispatch,
            state,
        );
        const inputWithFolderProps = getInputWithFolder().props();
        expect(inputWithFolderProps.value).toBe(undefined);
    });
});
