package com.xebialabs.xlrelease.delivery.upgrader

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliveryPersistence
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySchema.RELEASE_DELIVERIES
import com.xebialabs.xlrelease.domain.delivery.Transition
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.PersistenceSupport
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.codehaus.jettison.json.JSONObject
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component
import org.springframework.transaction.support.TransactionTemplate

import scala.collection.mutable

@Component
class XLRelease960DeliveryPropertiesUpgrader @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                          @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect,
                                                          @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                                          deliveryPersistence: DeliveryPersistence)
  extends Upgrade
    with PersistenceSupport
    with Logging
    with BatchSupport
    with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "9.6.0#2")

  override def doUpgrade(): Boolean = {
    doInBatch(getDeliveryIds, itemsName = "deliveries") { batch =>
      batch.items.foreach { deliveryId =>

        doInTransaction {
          deliveryPersistence.findById(deliveryId).foreach { deliveryRow =>
            val deliveryJson = new JSONObject(deliveryRow.content)
            // Move transitions inside stages and remove fromStageId field
            embedTransitionsInsideStages(deliveryJson)
            deliveryRow.deliveryId -> deliveryJson.toString()

            sqlExecWithContent(
              s"""|UPDATE ${RELEASE_DELIVERIES.TABLE}
                  | SET ${RELEASE_DELIVERIES.CONTENT} = :content
                  | WHERE ${RELEASE_DELIVERIES.ID} = :id""".stripMargin,
              params("id" -> getName(deliveryId)),
              "content" -> deliveryJson.toString(),
              _ => ()
            )
          }
        }
      }
    }

    true
  }

  private def getDeliveryIds: mutable.Seq[String] = {
    sqlQuery(
      s"SELECT del.${RELEASE_DELIVERIES.ID} FROM ${RELEASE_DELIVERIES.TABLE} del",
      params(),
      _.getString(RELEASE_DELIVERIES.ID)
    )
  }

  private def embedTransitionsInsideStages(deliveryJson: JSONObject): Unit = {
    val deliveryId = deliveryJson.getString("id")
    val transitions = deliveryJson.getJSONArray("transitions")
    val stagesJsonArray = deliveryJson.getJSONArray("stages")
    val stages = for {
      stageIdx <- 0 until stagesJsonArray.length()
    } yield stagesJsonArray.getJSONObject(stageIdx)

    for (transitionIdx <- 0 until transitions.length()) {
      val transitionJsonObj = transitions.getJSONObject(transitionIdx)

      transitionJsonObj.put("type", Type.valueOf(classOf[Transition]).toString)

      val fromStageId = Ids.getName(transitionJsonObj.remove("fromStageId").asInstanceOf[String])

      stages
        .find(stageObj => Ids.getName(stageObj.getString("id")) == fromStageId)
        .foreach { stageObj =>
          val stageId = stageObj.getString("id")
          // Reference to parent stage
          val transitionId = transitionJsonObj.getString("id")
          transitionJsonObj.put("id", transitionId.replace(deliveryId, stageId))
          transitionJsonObj.put("stage", stageId)

          stageObj.put("transition", transitionJsonObj)
        }
    }

    deliveryJson.remove("transitions")
  }
}
