package com.xebialabs.xlrelease.delivery.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySchema.{TRACKED_ITEMS => TIS}
import com.xebialabs.xlrelease.domain.delivery.TrackedItem
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

@Repository
@IsTransactional
class TrackedItemPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                          @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect,
                                          deliveryPersistence: DeliveryPersistence,
                                          folderPersistence: FolderPersistence) extends PersistenceSupport {
  private val STMT_INSERT_TRACKED_ITEM =
    s"""INSERT INTO ${TIS.TABLE}
       | ( ${TIS.ID}
       | , ${TIS.DELIVERY_UID}
       | , ${TIS.TITLE}
       | , ${TIS.IS_DESCOPED}
       | , ${TIS.CREATED_DATE}
       | , ${TIS.MODIFIED_DATE}
       | ) VALUES
       | ( :${TIS.ID}
       | , :${TIS.DELIVERY_UID}
       | , :${TIS.TITLE}
       | , :${TIS.IS_DESCOPED}
       | , :${TIS.CREATED_DATE}
       | , :${TIS.MODIFIED_DATE}
       | )
     """.stripMargin

  def insert(deliveryUid: CiUid, item: TrackedItem): CiUid = {
    try {
      sqlInsert(
        pkName(TIS.CI_UID),
        STMT_INSERT_TRACKED_ITEM,
        params(
          TIS.ID -> getName(item.getId),
          TIS.DELIVERY_UID -> deliveryUid,
          TIS.TITLE -> item.getTitle,
          TIS.IS_DESCOPED -> item.isDescoped.asInteger,
          TIS.CREATED_DATE -> item.getCreatedDate,
          TIS.MODIFIED_DATE -> item.getModifiedDate
        ),
        (ciUid: CiUid) => ciUid
      )
    } catch {
      case ex: DuplicateKeyException =>
        throw new IllegalArgumentException(s"Tracked Item with ID='${item.getId}' already exists", ex)

    }
  }

  def insertItems(deliveryUid: CiUid, trackedItems: Set[TrackedItem]): Unit = if (trackedItems.nonEmpty) {
    sqlBatch(
      STMT_INSERT_TRACKED_ITEM,
      trackedItems.map { item =>
        params(
          TIS.ID -> getName(item.getId),
          TIS.DELIVERY_UID -> deliveryUid,
          TIS.TITLE -> item.getTitle,
          TIS.IS_DESCOPED -> item.isDescoped.asInteger,
          TIS.CREATED_DATE -> item.getCreatedDate,
          TIS.MODIFIED_DATE -> item.getModifiedDate
        )
      }
    )
  }

  private val STMT_DELETE_TRACKED_ITEMS: String =
    s"""
       |DELETE FROM ${TIS.TABLE}
       | WHERE
       |  ${TIS.DELIVERY_UID} = :${TIS.DELIVERY_UID}
       |  AND ${TIS.ID} = :${TIS.ID}
       |""".stripMargin

  def deleteItems(deliveryUid: CiUid, itemIds: Set[CiId]): Unit = if (itemIds.nonEmpty) {
    sqlBatch(
      STMT_DELETE_TRACKED_ITEMS,
      itemIds.map { id =>
        Map(
          TIS.DELIVERY_UID -> deliveryUid,
          TIS.ID -> getName(id)
        )
      }
    )
  }

  private val STMT_UPDATE_TRACKED_ITEM: String =
    s"""
       |UPDATE ${TIS.TABLE}
       | SET
       |  ${TIS.TITLE} = :${TIS.TITLE}
       |  , ${TIS.IS_DESCOPED} = :${TIS.IS_DESCOPED}
       |  , ${TIS.MODIFIED_DATE} = :${TIS.MODIFIED_DATE}
       | WHERE
       |  ${TIS.ID} = :${TIS.ID}
       |""".stripMargin

  def updateTrackedItem(id: CiId, trackedItem: TrackedItem): Unit =
    sqlUpdate(
      STMT_UPDATE_TRACKED_ITEM,
      Map(
        TIS.TITLE -> trackedItem.getTitle,
        TIS.IS_DESCOPED -> trackedItem.isDescoped.asInteger,
        TIS.MODIFIED_DATE -> trackedItem.getModifiedDate,
        TIS.ID -> getName(id)
      ),
      _ => ()
    )
}
