package com.xebialabs.xlrelease.delivery.activity

import com.xebialabs.xlrelease.activity.ReleaseActivityUserFormatter._
import com.xebialabs.xlrelease.activity.{ActivityLogger, LoggingParams, ReleaseActivityLoggingEventHandler}
import com.xebialabs.xlrelease.delivery.activity.DeliveryActivity._
import com.xebialabs.xlrelease.delivery.events._
import com.xebialabs.xlrelease.delivery.transition.{ConditionTrigger, TransitionParams, UserTrigger}
import com.xebialabs.xlrelease.domain.ActivityLogEntry
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.{EventListener, Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.ActivityLogRepository
import com.xebialabs.xlrelease.repository.Ids.deliveryIdFrom
import com.xebialabs.xlrelease.service.{ReleaseService, UserInfoResolver}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.util.Try

@Component
@EventListener
class DeliveryActivityLoggingEventHandler @Autowired()(val eventBus: XLReleaseEventBus,
                                                       val activityLogRepository: ActivityLogRepository,
                                                       val releaseService: ReleaseService,
                                                       val userInfoResolver: UserInfoResolver)
  extends ActivityLogger[DeliveryEvent] {

  @Subscribe
  def onEvent(event: DeliveryEvent): Unit = log(event)

  override def logEntries: PartialFunction[XLReleaseEvent, LoggingParams] =
    (logDeliveryEntry orElse logStageEntry orElse logTrackedItemEntry orElse logTransitionEntry orElse logConditionEntry)
      .andThen(entry => LoggingParams(Some(deliveryIdFrom(entry.getTargetId)), Some(entry.getUsername), entry))

  private def logDeliveryEntry: PartialFunction[XLReleaseEvent, ActivityLogEntry] = {
    case ev@DeliveryCreatedEvent(delivery) => handleDeliveryEntry(ev, delivery, PATTERN_CREATED, DELIVERY_CREATED, delivery.getTitle)
    case ev@DeliveryUpdatedEvent(delivery) => handleDeliveryEntry(ev, delivery, PATTERN_UPDATED, DELIVERY_UPDATED, delivery.getTitle)
    case ev@DeliveryCreatedFromAsCodeEvent(delivery, scmTraceabilityData) => handleDeliveryEntry(ev, delivery, PATTERN_CREATED_FROM_AS_CODE,
      PATTERN_CREATED_FROM_AS_CODE, ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(scmTraceabilityData))
    case ev@DeliveryUpdatedFromAsCodeEvent(delivery, scmTraceabilityData) => handleDeliveryEntry(ev, delivery, PATTERN_UPDATED_FROM_AS_CODE,
      PATTERN_UPDATED_FROM_AS_CODE, ReleaseActivityLoggingEventHandler.getScmTraceabilityDataInfo(scmTraceabilityData))
    case ev@DeliveryDeletedEvent(delivery) => handleDeliveryEntry(ev, delivery, PATTERN_DELETED, DELIVERY_DELETED, delivery.getTitle)
    case ev@DeliveryCompletedEvent(delivery) => handleDeliveryEntry(ev, delivery, DELIVERY_COMPLETED, DELIVERY_COMPLETED, delivery.getTitle)
  }

  private def handleDeliveryEntry(event: DeliveryEvent,
                                  delivery: Delivery,
                                  patternActivity: DeliveryActivity,
                                  deliveryActivity: DeliveryActivity,
                                  parameters: Any*): ActivityLogEntry = {
    if (delivery.isTemplate) {
      patternActivity.create(event, delivery, parameters: _*)
    } else {
      deliveryActivity.create(event, delivery, parameters: _*)
    }
  }

  private def logStageEntry: PartialFunction[XLReleaseEvent, ActivityLogEntry] = {
    case ev@StageCreatedEvent(stage, delivery) => STAGE_CREATED.create(ev, stage, stage.getTitle)
    case ev@StageUpdatedEvent(original, updated, delivery) =>
      val fieldChanges = mutable.Buffer[String]()
      if (original.getTitle != updated.getTitle) {
        fieldChanges += s"Changed stage title from '${original.getTitle}' to '${updated.getTitle}'"
      }
      if (original.getOwner != updated.getOwner) {
        fieldChanges += s"Changed assigned user from ${quoteUser(original.getOwner, userInfoResolver)} to ${quoteUser(updated.getOwner, userInfoResolver)}"
      }
      if (original.getTeam != updated.getTeam) {
        fieldChanges += s"Changed assigned team from ${quoteAssignee(original.getTeam)} to ${quoteAssignee(updated.getTeam)}"
      }
      STAGE_UPDATED.create(ev, updated, updated.getTitle, fieldChanges.mkString(". "))
    case ev@StageRemovedEvent(stage, delivery) => STAGE_REMOVED.create(ev, stage, stage.getTitle)
    case ev@StageReopenedEvent(stage, delivery) => STAGE_REOPENED.create(ev, stage, stage.getTitle)
    case ev@StageStartedEvent(stage, delivery) => STAGE_STARTED.create(ev, stage, stage.getTitle)
    case ev@StageCompletedEvent(stage, delivery) => STAGE_COMPLETED.create(ev, stage, stage.getTitle)
  }

  //noinspection ScalaStyle
  private def logTrackedItemEntry: PartialFunction[XLReleaseEvent, ActivityLogEntry] = {
    case ev@ItemCreatedEvent(item, delivery) => ITEM_CREATED.create(ev, item, item.getTitle)
    case ev@ItemUpdatedEvent(item, delivery, oldItem) if item.getTitle != oldItem.getTitle =>
      ITEM_RENAMED.create(ev, item, oldItem.getTitle, item.getTitle)
    case ev@ItemRemovedEvent(item, delivery) => ITEM_REMOVED.create(ev, item, item.getTitle)
    case ev@ItemDescopedEvent(item, delivery) => ITEM_DESCOPED.create(ev, item, item.getTitle)
    case ev@ItemRescopedEvent(item, delivery) => ITEM_RESCOPED.create(ev, item, item.getTitle)
    case ev@ItemSkippedEvent(item, stage, delivery) => ITEM_SKIPPED.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle)
    case ev@ItemResetEvent(item, stage, delivery, fromReleaseId) if fromReleaseId != null =>
      ITEM_RESET.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle, Try(releaseService.getTitle(fromReleaseId)).getOrElse("Unknown"))
    case ev@ItemResetEvent(item, stage, delivery, _) => ITEM_RESET_MANUALLY.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle)
    case ev@ItemCompletedEvent(item, stage, delivery, fromReleaseId) if fromReleaseId != null =>
      ITEM_COMPLETED.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle, Try(releaseService.getTitle(fromReleaseId)).getOrElse("Unknown"))
    case ev@ItemCompletedEvent(item, stage, delivery, _) => ITEM_COMPLETED_MANUALLY.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle)
    case ev@ItemAvailableEvent(item, stage, delivery) => ITEM_AVAILABLE.create(ev, stage.getItemById(item.getId), item.getTitle, stage.getTitle)
    case ev@ItemTransitionApprovedEvent(item, stageItem, transition, stage, delivery) =>
      ITEM_TRANSITION_APPROVED.create(ev, stageItem, transition.getTitle, item.getTitle)
  }

  private def logTransitionEntry: PartialFunction[XLReleaseEvent, ActivityLogEntry] = {
    case ev@TransitionCreatedEvent(transition, delivery) => TRANSITION_CREATED.create(ev, transition, transition.getTitle)
    case ev@TransitionUpdatedEvent(original, updated, delivery) => TRANSITION_UPDATED.create(ev, original, original.getTitle)
    case ev@TransitionRemovedEvent(transition, delivery) => TRANSITION_REMOVED.create(ev, transition, transition.getTitle)
    case ev@TransitionExecutedEvent(transition, delivery, TransitionParams(_, _, _, _, UserTrigger(_))) =>
      TRANSITION_EXECUTED_MANUALLY.create(ev, transition, transition.getTitle)
    case ev@TransitionExecutedEvent(transition, delivery, TransitionParams(_, _, _, _, ConditionTrigger(_, _))) =>
      TRANSITION_EXECUTED_AUTOMATICALLY.create(ev, transition, transition.getTitle)
  }

  private def logConditionEntry: PartialFunction[XLReleaseEvent, ActivityLogEntry] = {
    case ev@ConditionSatisfiedEvent(condition, transition, _) if condition.isLeaf =>
      CONDITION_SATISFIED.create(ev, condition, "condition", condition.getDescription, transition.getTitle)
    case ev@ConditionUpdatedEvent(condition, transition, _) => CONDITION_UPDATED.create(ev, condition, condition.getDescription, transition.getTitle)
  }
}
