package com.xebialabs.xlrelease.delivery.repository.sql.persistence

import com.xebialabs.deployit.plumbing.serialization.MetadataIncludingCiConverter
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.SqlWithParameters
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySchema.{COLUMN_LENGTH_TITLE, RELEASE_DELIVERIES, RELEASE_DELIVERY_MEMBERS}
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySqlBuilder.STMT_RELEASE_DELIVERY_SELECT
import com.xebialabs.xlrelease.domain.delivery.{Delivery, DeliveryStatus}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{RichStringAsTruncatable, params}
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport, TenantId}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class DeliveryPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                       @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect,
                                       folderPersistence: FolderPersistence)
  extends PersistenceSupport {

  private val STMT_GET_UID_BY_ID = s"SELECT ${RELEASE_DELIVERIES.CI_UID} FROM ${RELEASE_DELIVERIES.TABLE} WHERE ${RELEASE_DELIVERIES.ID} = :id"

  def getUid(deliveryId: CiId): Option[CiUid] =
    sqlQuery(STMT_GET_UID_BY_ID, params("id" -> getName(deliveryId.normalized)), rs => CiUid(rs.getInt(RELEASE_DELIVERIES.CI_UID))).headOption

  val STMT_INSERT_RELEASE_DELIVERY: String =
    s"""|INSERT INTO ${RELEASE_DELIVERIES.TABLE}
        |   ( ${RELEASE_DELIVERIES.ID}
        |   , ${RELEASE_DELIVERIES.FOLDER_UID}
        |   , ${RELEASE_DELIVERIES.TITLE}
        |   , ${RELEASE_DELIVERIES.STATUS}
        |   , ${RELEASE_DELIVERIES.START_DATE}
        |   , ${RELEASE_DELIVERIES.END_DATE}
        |   , ${RELEASE_DELIVERIES.ORIGIN_PATTERN_ID}
        |   , ${RELEASE_DELIVERIES.CONTENT}
        |   , ${RELEASE_DELIVERIES.TENANT_ID}
        |   )
        | VALUES
        |   ( :${RELEASE_DELIVERIES.ID}
        |   , :${RELEASE_DELIVERIES.FOLDER_UID}
        |   , :${RELEASE_DELIVERIES.TITLE}
        |   , :${RELEASE_DELIVERIES.STATUS}
        |   , :${RELEASE_DELIVERIES.START_DATE}
        |   , :${RELEASE_DELIVERIES.END_DATE}
        |   , :${RELEASE_DELIVERIES.ORIGIN_PATTERN_ID}
        |   , :${RELEASE_DELIVERIES.CONTENT}
        |   , :${RELEASE_DELIVERIES.TENANT_ID}
        |   )
        """.stripMargin

  def insert(delivery: Delivery): CiUid = {
    val folderUid = folderPersistence.getUid(delivery.getFolderId)
    val content = serialize(delivery)
    try {
      sqlInsertWithContent(pkName(RELEASE_DELIVERIES.CI_UID), STMT_INSERT_RELEASE_DELIVERY, params(
        s"${RELEASE_DELIVERIES.ID}" -> getName(delivery.getId.normalized),
        s"${RELEASE_DELIVERIES.FOLDER_UID}" -> folderUid,
        s"${RELEASE_DELIVERIES.TITLE}" -> delivery.getTitle.truncate(COLUMN_LENGTH_TITLE),
        s"${RELEASE_DELIVERIES.STATUS}" -> delivery.getStatus.value(),
        s"${RELEASE_DELIVERIES.START_DATE}" -> delivery.getStartDate,
        s"${RELEASE_DELIVERIES.END_DATE}" -> delivery.getEndDate,
        s"${RELEASE_DELIVERIES.ORIGIN_PATTERN_ID}" -> getName(delivery.getOriginPatternId.normalized),
        s"${RELEASE_DELIVERIES.TENANT_ID}" -> delivery.getTenantId
      ), s"${RELEASE_DELIVERIES.CONTENT}" -> content,
        ciUid => ciUid
      )
    } catch {
      case ex: DuplicateKeyException => throw new IllegalArgumentException(s"Release delivery with ID '${delivery.getId}' already exists", ex)
    }
  }

  private val STMT_UPDATE_RELEASE_DELIVERY =
    s"""|UPDATE ${RELEASE_DELIVERIES.TABLE}
        | SET
        |  ${RELEASE_DELIVERIES.TITLE} = :newTitle,
        |  ${RELEASE_DELIVERIES.FOLDER_UID} = :folderUid,
        |  ${RELEASE_DELIVERIES.STATUS} = :newStatus,
        |  ${RELEASE_DELIVERIES.START_DATE} = :newStartDate,
        |  ${RELEASE_DELIVERIES.END_DATE} = :newEndDate,
        |  ${RELEASE_DELIVERIES.CONTENT} = :content
        | WHERE
        |  ${RELEASE_DELIVERIES.ID} = :id
        """.stripMargin

  def update(delivery: Delivery): Unit = {
    val folderUid = folderPersistence.getUid(delivery.getFolderId)
    val content = serialize(delivery)
    sqlExecWithContent(STMT_UPDATE_RELEASE_DELIVERY, params(
      "id" -> getName(delivery.getId.normalized),
      "folderUid" -> folderUid,
      "newTitle" -> delivery.getTitle.truncate(COLUMN_LENGTH_TITLE),
      "newStatus" -> delivery.getStatus.value(),
      "newStartDate" -> delivery.getStartDate,
      "newEndDate" -> delivery.getEndDate
    ), "content" -> content,
      checkCiUpdated(delivery.getId)
    )
  }

  private val STMT_DELETE_RELEASE_DELIVERY_BY_ID =
    s"""|DELETE FROM ${RELEASE_DELIVERIES.TABLE}
        | WHERE ${RELEASE_DELIVERIES.ID} = :id""".stripMargin

  def delete(deliveryId: CiId): Unit = sqlExec(STMT_DELETE_RELEASE_DELIVERY_BY_ID, params("id" -> getName(deliveryId.normalized)), ps => Try(ps.execute()))

  private val STMT_EXISTS_RELEASE_DELIVERY_BY_ID: String = s"SELECT COUNT(*) FROM ${RELEASE_DELIVERIES.TABLE} WHERE ${RELEASE_DELIVERIES.ID} = :id"

  def exists(deliveryId: CiId): Boolean = sqlQuery(STMT_EXISTS_RELEASE_DELIVERY_BY_ID, params("id" -> getName(deliveryId.normalized)), _.getInt(1) > 0).head

  private val STMT_FIND_RELEASE_DELIVERY_BY_ID =
    s"""|$STMT_RELEASE_DELIVERY_SELECT
        | WHERE del.${RELEASE_DELIVERIES.ID} = ?""".stripMargin

  def findById(id: CiId): Option[DeliveryRow] = {
    findOptional(_.queryForObject(STMT_FIND_RELEASE_DELIVERY_BY_ID, deliveryRowMapper, getName(id.normalized)))
  }

  private val STMT_FIND_RELEASE_DELIVERY_BY_TITLE =
    s"""|$STMT_RELEASE_DELIVERY_SELECT
        | WHERE LOWER(del.${RELEASE_DELIVERIES.TITLE}) = ?""".stripMargin

  def findByTitle(title: String): Option[DeliveryRow] = {
    findOptional(_.queryForObject(STMT_FIND_RELEASE_DELIVERY_BY_TITLE, deliveryRowMapper, title.toLowerCase))
  }

  def findByQuery(sqlWithParameters: SqlWithParameters): Seq[DeliveryRow] = {
    val (sql, params) = sqlWithParameters
    jdbcTemplate.query[DeliveryRow](sql, deliveryRowMapper, params: _*).asScala.toSeq
  }

  private val STMT_GET_FOLDER_ID_BY_DELIVERY_ID =
    s"""| SELECT f.${FOLDERS.FOLDER_ID}
        | FROM ${RELEASE_DELIVERIES.TABLE} del
        | INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = del.${RELEASE_DELIVERIES.FOLDER_UID}
        | WHERE del.${RELEASE_DELIVERIES.ID} = :deliveryId""".stripMargin

  def findFolderId(deliveryId: String): FolderId = {
    sqlQuery(STMT_GET_FOLDER_ID_BY_DELIVERY_ID, params("deliveryId" -> getName(deliveryId.normalized)),
      rs => Option(rs.getString(1)).map(FolderId(_)).getOrElse(FolderId.Root)
    ).headOption.getOrElse(throw new LogFriendlyNotFoundException(s"Cannot find folderId for non existing delivery $deliveryId"))
  }

  private val STMT_GET_RELEASES_BY_DELIVERY_ID =
    s"""|SELECT rel.${RELEASES.RELEASE_ID}, rel.${RELEASES.RELEASE_TITLE}, rel.${RELEASES.STATUS}, rel.${RELEASES.START_DATE},
        | rel.${RELEASES.END_DATE}
        | FROM ${RELEASES.TABLE} rel
        | LEFT JOIN ${RELEASE_DELIVERY_MEMBERS.TABLE} m ON m.${RELEASE_DELIVERY_MEMBERS.RELEASE_ID} = rel.${RELEASES.RELEASE_ID}
        | INNER JOIN ${RELEASE_DELIVERIES.TABLE} d ON d.${RELEASE_DELIVERIES.CI_UID} = m.${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID}
        | WHERE d.${RELEASE_DELIVERIES.ID} = :deliveryId""".stripMargin

  def findReleasesByDeliveryId(deliveryId: String): Seq[ReleaseBasicDataExt] = {
    sqlQuery(STMT_GET_RELEASES_BY_DELIVERY_ID, params("deliveryId" -> getName(deliveryId.normalized)),
      rs => ReleaseBasicDataExt(
        id = rs.getString(RELEASES.RELEASE_ID),
        title = rs.getString(RELEASES.RELEASE_TITLE),
        status = rs.getString(RELEASES.STATUS),
        startDate = rs.getTimestamp(RELEASES.START_DATE),
        endDate = rs.getTimestamp(RELEASES.END_DATE)
      )).toSeq
  }

  private val STMT_FIND_PATTERN_BY_TITLE =
    s"""|$STMT_RELEASE_DELIVERY_SELECT
        | WHERE LOWER(del.${RELEASE_DELIVERIES.TITLE}) = ?
        | AND LOWER(del.${RELEASE_DELIVERIES.STATUS}) = '${DeliveryStatus.TEMPLATE.value()}'""".stripMargin

  def findPatternByTitle(title: String): Option[DeliveryRow] = {
    findOptional(_.queryForObject(STMT_FIND_PATTERN_BY_TITLE, deliveryRowMapper, title.toLowerCase))
  }

  private val STMT_COUNT_NO_OF_TENANT_DELIVERIES: String =
    s"""|SELECT COUNT(1)
        | FROM ${RELEASE_DELIVERIES.TABLE}
        | WHERE ${RELEASE_DELIVERIES.STATUS} = :${RELEASE_DELIVERIES.STATUS}
        | AND ${RELEASE_DELIVERIES.TENANT_ID} = :${RELEASE_DELIVERIES.TENANT_ID}
        |""".stripMargin

  def tenantDeliveryCount(tenantId: TenantId, status: DeliveryStatus): Integer = {
    namedTemplate.queryForObject(STMT_COUNT_NO_OF_TENANT_DELIVERIES,
      paramSource(RELEASE_DELIVERIES.TENANT_ID -> tenantId, RELEASE_DELIVERIES.STATUS -> status.value),
      classOf[Integer])
  }


  private val deliveryRowMapper: RowMapper[DeliveryRow] = (rs: ResultSet, _: Int) => {
    val contentStream = rs.getBinaryStream(RELEASE_DELIVERIES.CONTENT)
    try {
      val content = decompress(contentStream)
      DeliveryRow(
        content,
        ciUid = rs.getInt(RELEASE_DELIVERIES.CI_UID),
        folderId = FolderId(rs.getString(FOLDERS.FOLDER_PATH)) / rs.getString(FOLDERS.FOLDER_ID),
        deliveryId = rs.getString(RELEASE_DELIVERIES.ID),
        tenantId = rs.getString(RELEASE_DELIVERIES.TENANT_ID)
      )
    } finally {
      contentStream.close()
    }
  }

  private def serialize(delivery: Delivery): String = {
    val ciConverter = new MetadataIncludingCiConverter(null, null)
    val deliveryCopy = cloneCi(delivery)
    deliveryCopy.setFolderId(null)
    val content = CiSerializerHelper.serialize(deliveryCopy, ciConverter)
    content
  }
}
