package com.xebialabs.xlrelease.delivery.events.handler

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.delivery.events._
import com.xebialabs.xlrelease.domain.delivery.TrackedItemStatus.{NOT_READY, READY}
import com.xebialabs.xlrelease.domain.delivery.{Delivery, StageCompletionSubscriber, TrackedItemStatus, TrackedItemStatusChangeSubscriber}
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.repository.query.TaskBasicData
import com.xebialabs.xlrelease.user.User.SYSTEM
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@EventListener
class DeliveryLifecycleSubscriberHandler @Autowired()(releaseActorService: ReleaseActorService, taskRepository: TaskRepository) {
  @Subscribe
  def onDeliveryDeleted(event: DeliveryDeletedEvent): Unit = {
    val taskIds = event.delivery.getSubscribers.asScala.map(_.getSourceId).distinct
    taskRepository.findTasksBasicData(taskIds.asJava).forEach { task =>
      if (task.status == TaskStatus.IN_PROGRESS) {
        releaseActorService.failTask(task.taskId, s"Unable to complete task, delivery '${event.delivery.getTitle}' was deleted from the system", SYSTEM)
      }
    }
  }
}

@Component
@EventListener
class StageCompletionSubscriberHandler @Autowired()(releaseActorService: ReleaseActorService, taskRepository: TaskRepository) {
  @Subscribe
  def onStageCompleted(event: StageCompletedEvent): Unit = {
    val taskIds = event.delivery.getSubscribersOfType(classOf[StageCompletionSubscriber]).asScala
      .collect { case sub if sub.stageId == event.stage.getId => sub.sourceId }.distinct
    val tasks = taskRepository.findTasksBasicData(taskIds.asJava)
    tasks.forEach { task =>
      if (task.status == TaskStatus.IN_PROGRESS) {
        releaseActorService.markTaskAsDone(TaskStatus.COMPLETED, task.taskId, s"Stage '${event.stage.getTitle}' completed", SYSTEM)
      }
    }
  }
}

@Component
@EventListener
class TrackedItemStatusSubscriberHandler @Autowired()(releaseActorService: ReleaseActorService, taskRepository: TaskRepository) {

  @Subscribe
  def onItemCompleted(event: ItemCompletedEvent): Unit = {
    getActiveItemSubscribers(event.delivery, event.item.getId, Some(event.stage.getId), Some(READY)).foreach { task =>
      releaseActorService.markTaskAsDone(TaskStatus.COMPLETED, task.taskId,
        s"All tracked items are completed in stage '${event.stage.getTitle}'", SYSTEM)
    }
  }

  @Subscribe
  def onItemAvailable(event: ItemAvailableEvent): Unit = {
    getActiveItemSubscribers(event.delivery, event.item.getId, Some(event.stage.getId), Some(NOT_READY)).foreach { task =>
      releaseActorService.markTaskAsDone(TaskStatus.COMPLETED, task.taskId,
        s"All tracked items are available in stage '${event.stage.getTitle}'", SYSTEM)
    }
  }

  @Subscribe
  def onItemSkipped(event: ItemSkippedEvent): Unit = {
    getActiveItemSubscribers(event.delivery, event.item.getId, Some(event.stage.getId)).foreach { task =>
      releaseActorService.failTask(task.taskId,
        s"Unable to complete task, tracked item '${event.item.getTitle}' was skipped in stage '${event.stage.getTitle}'", SYSTEM)
    }
  }

  @Subscribe
  def onItemDescoped(event: ItemDescopedEvent): Unit = {
    getActiveItemSubscribers(event.delivery, event.item.getId).foreach { task =>
      releaseActorService.failTask(task.taskId, s"Unable to complete task, tracked item '${event.item.getTitle}' was descoped", SYSTEM)
    }
  }

  @Subscribe
  def onItemRemoved(event: ItemRemovedEvent): Unit = {
    getActiveItemSubscribers(event.delivery, event.item.getId).foreach { task =>
      releaseActorService.failTask(task.taskId, s"Unable to complete task, tracked item '${event.item.getTitle}' was removed", SYSTEM)
    }
  }

  private def getActiveItemSubscribers(delivery: Delivery, itemId: String,
                                       stageId: Option[String] = None, status: Option[TrackedItemStatus] = None): Seq[TaskBasicData] = {
    val taskIds = delivery.getSubscribersOfType(classOf[TrackedItemStatusChangeSubscriber]).asScala
      .collect {
        case sub if sub.matches(delivery, itemId, stageId, status) => sub.sourceId
      }.distinct
    taskRepository.findTasksBasicData(taskIds.asJava).asScala.filter(_.status == TaskStatus.IN_PROGRESS).toSeq
  }
}
