package com.xebialabs.xlrelease.delivery.actors

import com.xebialabs.xlrelease.actors.{ActorSystemHolder, ManagedActor}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.delivery.actors.DeliveryActor.DeliveryAction
import com.xebialabs.xlrelease.delivery.actors.DeliveryConfiguration.DeliveryActorHolder
import com.xebialabs.xlrelease.delivery.service.{DeliveryExecutionService, DeliveryPatternService, DeliveryService}
import org.apache.pekko.cluster.sharding.ShardRegion
import org.springframework.context.annotation.{Bean, Configuration}

object DeliveryConfiguration {
  type DeliveryActorHolder = ManagedActor[DeliveryActor]
}
@Configuration
class DeliveryConfiguration(xlrConfig: XlrConfig,
                            actorSystemHolder: ActorSystemHolder,
                            deliveryService: DeliveryService,
                            deliveryPatternService: DeliveryPatternService,
                            deliveryExecutionService: DeliveryExecutionService) {
  @Bean
  def deliveryActorHolder: DeliveryActorHolder = {
    val deliveryActorProps = DeliveryActor.props(clustered = xlrConfig.isClusterEnabled, deliveryService, deliveryPatternService, deliveryExecutionService)

    if (xlrConfig.isClusterEnabled) {
      val numberOfShards = xlrConfig.sharding.numberOfReleaseShards

      def extractShardId(id: String): String = (math.abs(id.deliveryActorName.hashCode) % numberOfShards).toString

      val extractDeliveryId: ShardRegion.ExtractEntityId = {
        case msg: DeliveryAction => (msg.deliveryId.deliveryActorName, msg)
      }

      val extractDeliveryShardId: ShardRegion.ExtractShardId = {
        case msg: DeliveryAction => extractShardId(msg.deliveryId)
        case ShardRegion.StartEntity(id) => extractShardId(id)
      }

      actorSystemHolder.shardedActorOf(
        deliveryActorProps,
        "Delivery",
        extractDeliveryId,
        extractDeliveryShardId
      )
    } else {
      val releaseDeliveryActorMaker: DeliveryActorMaker = (ctx, deliveryId) => {
        ctx.actorOf(
          deliveryActorProps,
          deliveryId.deliveryActorName
        )
      }

      val deliveryProcessingActorProps = DeliveryProcessingActor.props(releaseDeliveryActorMaker)
      actorSystemHolder.actorOf(deliveryProcessingActorProps, DeliveryProcessingActor.name)
    }
  }

  @Bean
  def deliveryActorService(): DeliveryActorService = {
    new DeliveryActorService(deliveryActorHolder)
  }
}
