package com.xebialabs.xlrelease.delivery.service

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.xlrelease.api.v1.forms.{CreateDelivery, DeliveryPatternFilters, DuplicateDeliveryPattern}
import com.xebialabs.xlrelease.db.{ArchivedReleases, DbConstants}
import com.xebialabs.xlrelease.delivery.events._
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissions.VIEW_DELIVERY_PATTERN_PERMISSION_SET
import com.xebialabs.xlrelease.domain.delivery.Stage.DEFAULT_STAGE_TITLE
import com.xebialabs.xlrelease.domain.delivery._
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.exception.LogFriendlyConcurrentModificationException
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids, Page, ReleaseRepository}
import com.xebialabs.xlrelease.service.{CiIdService, FolderService, ReleaseService}
import io.micrometer.core.annotation.Timed
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util
import java.util.{Collections, Date, Optional}
import scala.jdk.CollectionConverters._
import scala.reflect.ClassTag

@Service
class DeliveryPatternService @Autowired()(val deliveryRepository: DeliveryRepository,
                                          val releaseRepository: ReleaseRepository,
                                          val releaseService: ReleaseService,
                                          val ciIdService: CiIdService,
                                          val archivedReleases: ArchivedReleases,
                                          val folderService: FolderService,
                                          val roleService: RoleService,
                                          val eventBus: XLReleaseEventBus)
  extends DeliveryServiceUtils {

  @Timed
  def getPattern(patternId: String): Delivery = {
    val pattern = getDeliveryOrPattern(patternId)
    checkIsPattern(pattern)
    pattern
  }

  @Timed
  def getPatternByIdOrTitle(patternIdOrTitle: String): Delivery = {
    val pattern = deliveryRepository.getByIdOrTitle(patternIdOrTitle)
    checkIsPattern(pattern)
    pattern
  }

  // If title exists, and ID is null => true
  // If title exists, ID is not null and ID == the ID of the pattern found by title => then true
  @Timed
  def existsPatternWithTitle(id: String, title: String): Boolean = {
    alreadyExistsByTitle(title, id)
  }

  @Timed
  def existsPattern(deliveryId: String): Boolean = deliveryRepository.exists(deliveryId)


  @Timed
  def createDeliveryPattern(pattern: Delivery): Delivery = createDeliveryPattern(pattern, shouldResetPattern = true)

  @Timed
  def createDeliveryPattern(pattern: Delivery, shouldResetPattern: Boolean): Delivery = {
    checkNotNull(pattern, "Release delivery")
    logger.debug(s"Creating new delivery pattern '$pattern'")
    pattern.setStatus(DeliveryStatus.TEMPLATE)
    validatePattern(pattern)

    if (shouldResetPattern) {
      resetPattern(pattern)
    }

    if (pattern.getStages.isEmpty) {
      val defaultStage = new Stage(DEFAULT_STAGE_TITLE)
      defaultStage.setId(factory.stageId(pattern.getId))
      pattern.addStage(defaultStage)
    }

    validateStagesAndTransitionsAndItems(pattern)

    deliveryRepository.create(pattern)
    eventBus.publish(DeliveryCreatedEvent(pattern))

    pattern
  }

  @Timed
  def updateDeliveryPattern(updated: Delivery): Delivery = {
    logger.debug(s"Updating delivery pattern '$updated'")
    validatePattern(updated)
    doUpdate(updated)
  }

  @Timed
  def deleteDeliveryPattern(patternId: String): Unit = {
    logger.debug(s"Deleting release delivery pattern '$patternId'")
    val pattern = deliveryRepository.read(patternId)
    checkIsPattern(pattern)
    doDelete(patternId)
  }

  @Timed
  def duplicateDeliveryPattern(patternId: String, params: DuplicateDeliveryPattern): Delivery = {
    logger.debug(s"Duplicating delivery pattern '$patternId' with parameters '$params'")
    val pattern = getPattern(patternId)

    resetPattern(pattern, true)
    if (params.hasDescription) {
      pattern.setDescription(params.getDescription)
    }
    if (params.hasTitle) {
      pattern.setTitle(params.getTitle)
    } else {
      val regExp = raw"""\(([0-9]+)\)$$""".r

      val c = regExp.split(pattern.getTitle)
      if (c.head.length == pattern.getTitle.length) {
        pattern.setTitle(s"${pattern.getTitle} (0)")
      }

      regExp.findFirstIn(pattern.getTitle) match {
        case Some(value) => {
          var orderNumber = value.substring(1, value.length - 1).toInt
          do {
            orderNumber += 1
            val newTitle = regExp.replaceAllIn(pattern.getTitle, s"(${orderNumber.toString})")
            pattern.setTitle(newTitle)
          } while (alreadyExistsByTitle(pattern.getTitle, pattern.getId) && orderNumber < 100)
        }
        case None =>
      }
    }

    validatePattern(pattern)

    deliveryRepository.create(pattern)
    val duplicatedPattern = deliveryRepository.read(pattern.getId)
    eventBus.publish(DeliveryCreatedEvent(duplicatedPattern))
    duplicatedPattern
  }

  @Timed
  def createDeliveryFromPattern(patternId: String, parameters: CreateDelivery): Delivery = {
    logger.debug(s"Creating new delivery from pattern '$patternId' with parameters '$parameters'")

    val delivery = getPattern(patternId)

    resetPattern(delivery)

    delivery.setTitle(parameters.getTitle)
    delivery.setDescription(parameters.getDescription)
    delivery.setOriginPatternId(patternId)
    delivery.setAutoComplete(parameters.isAutoComplete)
    if (parameters.getFolderId != null) {
      delivery.setFolderId(parameters.getFolderId)
    }

    computeAndUpdateDates(delivery, parameters.getDuration, parameters.getStartDate, parameters.getEndDate)

    checkArgument(delivery.getStages != null && !delivery.getStages.isEmpty, "Cannot create delivery from pattern without stages")

    val manager = DeliveryStateManager(delivery, ciIdService)
    val startedDelivery = manager.start()

    validateDelivery(startedDelivery)

    eventBus.publish(DeliveryCreatingEvent(startedDelivery))
    deliveryRepository.create(startedDelivery)
    eventBus.publish(DeliveryCreatedEvent(startedDelivery))
    manager.getEvents.foreach(eventBus.publish)

    startedDelivery
  }

  @Timed
  def searchPatterns(filters: DeliveryPatternFilters, page: Page, enforcePermissions: Boolean = true): util.List[Delivery] =
    deliveryRepository.search(
      filters,
      page,
      null,
      principals = currentPrincipals,
      roleIds = currentRoleIds,
      anyOfPermissions = Seq(VIEW_DELIVERY_PATTERN_PERMISSION_SET: _*),
      enforcePermissions).asJava

  // STAGES

  @Timed
  def getStages(patternId: String): java.util.List[Stage] = getPattern(patternId).getStages

  @Timed
  def addStage(patternId: String, stage: Stage, position: Optional[Integer] = Optional.empty[Integer]): Stage = {
    addStage(getPattern(patternId), stage, position)
  }

  @Timed
  def addStageBetween(patternId: String, stage: Stage, before: Option[String], after: Option[String]): Stage = {
    val pattern: Delivery = getPattern(patternId)

    def getStageIndex = (id: String) => {
      val folderlessId = Ids.getName(id)
      pattern.getStages.asScala.zipWithIndex.find(t => Ids.getName(t._1.getId).equals(folderlessId)).getOrElse(throw new LogFriendlyConcurrentModificationException("Stage %s not found", id))._2
    }

    val beforeIndex = before.map(getStageIndex)
    val afterIndex = after.map(getStageIndex)
    if (before.isDefined) {
      if (after.isDefined) {
        // insert between
        if (afterIndex.get == beforeIndex.get - 1) {
          addStage(pattern, stage, Optional.of[Integer](beforeIndex.get))
        } else {
          throw new LogFriendlyConcurrentModificationException("Unable to add stage between %s and %s, because these two stages are not neighbors any more", after.get, before.get)
        }
      } else {
        // add to the beginning of the list
        if (beforeIndex.get == 0) {
          addStage(pattern, stage, Optional.of[Integer](0))
        } else {
          throw new LogFriendlyConcurrentModificationException("Unable to add first stage before %s, because it is not first stage of delivery pattern any more", before.get)
        }
      }
    } else {
      if (after.isDefined) {
        // add to the end of list, where list is not empty
        if (afterIndex.get == pattern.getStages.size() - 1) {
          addStage(pattern, stage, Optional.empty[Integer])
        } else {
          throw new LogFriendlyConcurrentModificationException("Unable to add last stage after %s, because it is not last stage of the delivery pattern any more", after.get)
        }
      } else {
        // neither before, nor after is defined, so list is expected to be empty, add new item
        if (pattern.getStages.isEmpty) {
          addStage(pattern, stage, Optional.empty[Integer])
        } else {
          throw new LogFriendlyConcurrentModificationException("Unable to add first stage to the pattern, because pattern is not empty any more")
        }
      }
    }
  }

  @Timed
  def addStage(pattern: Delivery, stage: Stage, position: Optional[Integer]): Stage = {
    logger.debug(s"Adding new stage '$stage' to pattern '${pattern.getId}'")

    checkIsUpdatable(pattern)

    val stages = pattern.getStages.asScala.toSeq
    generateTitleIfNecessary(stage, stages)
    val realPosition = position.orElse(stages.size)
    validateStageProperties(stage)
    validateNewStage(stages, stage, realPosition)

    stage.setId(factory.stageId(pattern.getId))
    stage.setStatus(StageStatus.OPEN)
    stage.setTransition(null)
    stage.setItems(Collections.emptyList())

    pattern.addStage(stage, realPosition)

    deliveryRepository.update(pattern)
    eventBus.publish(StageCreatedEvent(stage, pattern))
    stage
  }

  @Timed
  def updateStage(patternId: String, updated: Stage, fromBatch: Boolean = false): Stage = {
    logger.debug(s"Updating stage '$updated' on pattern '$patternId'")

    val pattern = getPattern(patternId)
    checkIsUpdatable(pattern)

    validateStageProperties(updated)
    val stages = pattern.getStages.asScala.toSeq
    val original = pattern.getStageByIdOrTitle(updated.getId)
    val originalBeforeChanges = CiCloneHelper.cloneCi(original)
    if (!fromBatch) {
      validateNewStage(stages diff Seq(original), updated, stages.indexOf(original))
    }

    original.setTitle(updated.getTitle)
    original.setOwner(updated.getOwner)
    original.setTeam(updated.getTeam)

    deliveryRepository.update(pattern)
    eventBus.publish(StageUpdatedEvent(originalBeforeChanges, original, pattern))
    original
  }

  @Timed
  def deleteStage(patternId: String, stageId: String): Unit = {
    logger.debug(s"Removing stage '$stageId' from pattern '$patternId'")

    val pattern = getPattern(patternId)
    checkIsUpdatable(pattern)

    val stage = pattern.getStageByIdOrTitle(stageId)
    if (pattern.isLastStage(stage)) {
      pattern.findPreviousStage(stage).ifPresent(_.setTransition(null))
    }
    pattern.removeStage(stage)

    deliveryRepository.update(pattern)
    eventBus.publish(StageRemovedEvent(stage, pattern))
  }

  // TRANSITIONS

  @Timed
  def addTransition(patternId: String, stageIdOrTitle: String, transition: Transition): Transition = {
    addTransition(getPattern(patternId), stageIdOrTitle, transition)
  }

  @Timed
  def addTransition(pattern: Delivery, stageIdOrTitle: String, transition: Transition): Transition = {
    logger.debug(s"Adding new transition '$transition' to pattern '${pattern.getId}'")

    checkIsUpdatable(pattern)

    val stage = pattern.getStageByIdOrTitle(stageIdOrTitle)
    validateTransition(pattern, stage, transition)
    if (stage.getTransition != null) {
      throw new LogFriendlyConcurrentModificationException("There is already a transition named '%s' associated with stage '%s'", stage.getTransition.getTitle, stage.getTitle)
    }

    transition.setId(factory.transitionId(stage.getId))
    transition.getAllConditions.forEach(resetCondition(transition, _))
    stage.setTransition(transition)

    deliveryRepository.update(pattern)
    eventBus.publish(TransitionCreatedEvent(transition, pattern))
    transition
  }

  @Timed
  def updateTransition(patternId: String, updated: Transition): Transition = {
    updateTransition(getPattern(patternId), updated)
  }

  @Timed
  def updateTransition(pattern: Delivery, updated: Transition): Transition = {
    logger.debug(s"Updating transition '$updated' on pattern '${pattern.getId}'")

    checkIsUpdatable(pattern)
    checkNotNull(updated, "Transition")

    val stage = pattern.getStageByTransition(updated)
    val original = stage.getTransition
    validateTransition(pattern, stage, updated)

    updated.getAllConditions.forEach(resetCondition(updated, _))
    stage.setTransition(updated)

    deliveryRepository.update(pattern)
    eventBus.publish(TransitionUpdatedEvent(original, updated, pattern))
    updated
  }

  @Timed
  def deleteTransition(patternId: String, transitionId: String): Unit = {
    logger.debug(s"Removing transition '$transitionId' from pattern '$patternId'")

    val pattern = getPattern(patternId)
    checkIsUpdatable(pattern)

    val transition = pattern.getTransitionByIdOrTitle(transitionId)
    pattern.getStageByTransition(transition).setTransition(null)

    deliveryRepository.update(pattern)
    eventBus.publish(TransitionRemovedEvent(transition, pattern))
  }

  // PRIVATE

  private def generateTitleIfNecessary(stage: Stage, stages: Seq[Stage]): Unit = {
    checkNotNull(stage, "Stage")

    if (stage.getTitle == null) {
      var index = 0

      def title(): String =
        if (index > 0) {
          s"$DEFAULT_STAGE_TITLE ($index)"
        } else {
          DEFAULT_STAGE_TITLE
        }

      while (stages.exists(_.getTitle.trim.toLowerCase == title().toLowerCase)) {
        index += 1
      }

      stage.setTitle(title())
    }
  }

  private def checkIsPattern(delivery: Delivery): Unit = {
    checkArgument(delivery.isTemplate, s"${delivery.getId} must be a Delivery Pattern")
  }

  def validatePattern(delivery: Delivery): Unit = {
    validatePattern(delivery, true)
  }

  def validatePattern(delivery: Delivery, withQueries: Boolean): Unit = {
    validate(delivery, withQueries)
    checkIsPattern(delivery)
    checkArgument(delivery.getPlannedDuration <= 73755, "Duration must be equal or less than 99 months 99 days 99 hours (73755 hours)")
    if (withQueries) checkArgument(!alreadyExistsByTitle(delivery.getTitle, delivery.getId), s"Pattern with title '${delivery.getTitle}' already exists")
  }

  // Validation for deep create
  def validateStagesAndTransitionsAndItems(pattern: Delivery): Unit = {
    pattern.getStages.forEach { stage =>
      validateStageProperties(stage)
      Option(stage.getTransition).foreach(t => validateTransition(pattern, stage, t))
    }
    val stageTitles = pattern.getStages.asScala.map(_.getTitle)
    val duplicateStages = stageTitles.diff(stageTitles.distinct).distinct
    checkArgument(duplicateStages.isEmpty, s"Stages with duplicate titles [${duplicateStages.mkString(", ")}] are not allowed in pattern")

    val trackedItemTitles = pattern.getTrackedItems.asScala.map(_.getTitle.toLowerCase())
    val duplicateItems = trackedItemTitles.diff(trackedItemTitles.distinct).distinct
    checkArgument(duplicateItems.isEmpty, s"Tracked items with duplicate titles [${duplicateItems.mkString(", ")}] are not allowed in pattern")
  }

  private def validateNewStage(stages: Seq[Stage], stage: Stage, position: Int): Unit = {
    checkArgument(position >= 0 && position <= stages.size, "Stage index out of bounds")
    checkArgument(!stages.exists(_.getTitle == stage.getTitle), "Stage title already exists in delivery pattern")
    // in the future validate stage conditions here
  }

  private def validateStageProperties(stage: Stage): Unit = {
    checkNotNull(stage, "Stage")
    checkArgument(isNotBlank(stage.getTitle), "Stage title must be set")
    checkArgument(stage.getTitle.length < 256, "Stage title must be 255 characters or less")
    checkArgument(!stage.getTitle.contains(DbConstants.ESCAPE_CHAR), s"Title must not contain reserved character ${DbConstants.ESCAPE_CHAR}")
  }

  def resetPattern(pattern: Delivery, forceNewIds: Boolean = false): Unit = {
    val resetDate = new Date()

    val oldDeliveryId = pattern.getId
    val newDeliveryId = factory.deliveryId()

    pattern.setId(newDeliveryId)

    pattern.getStages.forEach { stage =>
      val oldStageId = stage.getId
      val newStageId = generateOrUpdateId[Stage](oldStageId, oldDeliveryId, newDeliveryId, forceNewIds)

      stage.setId(newStageId)
      Option(stage.getTransition).foreach(transition => {
        val oldTransitionId = transition.getId
        val newTransitionId = generateOrUpdateId[Transition](transition.getId, oldStageId, newStageId, forceNewIds)

        transition.setId(newTransitionId)
        transition.setStage(stage)
        transition.getAllConditions.forEach { condition =>
          condition.setId(generateOrUpdateId[Condition](condition.getId, oldTransitionId, newTransitionId, forceNewIds))
        }
      })
    }

    pattern.getTrackedItems.forEach { item =>
      // Generate Item Ids
      item.setId(factory.trackedItemId(newDeliveryId))
      item.setCreatedDate(resetDate)
      item.setModifiedDate(resetDate)
    }
  }

  private def resetCondition(transition: Transition, condition: Condition): Unit = {
    condition.setId(generateOrUpdateId[Condition](condition.getId, transition.getId, transition.getId, forceNewIds = false))
    condition.reset()
  }

  private def generateOrUpdateId[T: ClassTag](existingId: String, oldParentId: String, newParentId: String, forceNewIds: Boolean): String = {
    val typeName = Type.valueOf(implicitly[ClassTag[T]].runtimeClass).getName
    val idNotDefined = existingId == null || existingId.trim.isEmpty || !getName(existingId).startsWith(typeName)
    if (forceNewIds || idNotDefined) {
      factory.createUniqueId[T](newParentId)
    } else {
      existingId.replace(Option(oldParentId).getOrElse(""), Option(newParentId).getOrElse(""))
    }
  }

  private def alreadyExistsByTitle(patternTitle: String, patternId: String = null): Boolean = {
    val filters = new DeliveryPatternFilters()
    filters.withTitle(patternTitle)
    filters.withStrictTitleMatch(true)
    deliveryRepository.searchIds(filters).exists(deliveryId => getName(deliveryId) != getName(patternId))
  }

  override protected def checkIsUpdatable(existingDelivery: Delivery, action: String): Unit = existingDelivery.isTemplate()
}
