package com.xebialabs.xlrelease.delivery.security

import com.xebialabs.deployit.security.PermissionDeniedException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissions._
import com.xebialabs.xlrelease.delivery.service.{DeliveryPatternService, DeliveryService}
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.utils.DeliveryUtils.deliveryIdFrom
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_DELIVERY
import com.xebialabs.xlrelease.security.authentication.AuthenticationService
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

@Component
class DeliveryPermissionChecker @Autowired()(
                                              authenticationService: AuthenticationService,
                                              deliveryPatternService: DeliveryPatternService,
                                              deliveryService: DeliveryService,
                                              permissions: PermissionChecker
                                            ) {
  def checkView(deliveryId: String): Unit = {
    val delivery = deliveryService.getDeliveryOrPattern(deliveryId)
    if (delivery.isTemplate) {
      checkViewDeliveryPattern(delivery)
    } else {
      checkViewDelivery(delivery)
    }
  }

  def checkViewDelivery(deliveryId: String): Unit = {
    val folderId = deliveryService.getFolderId(deliveryId)
    checkViewDeliveryOnFolder(folderId)
  }

  def checkViewDelivery(delivery: Delivery): Unit = {
    val folderId = delivery.getFolderId
    checkViewDeliveryOnFolder(folderId)
  }

  def checkEditDelivery(deliveryId: String): Unit = {
    val folderId = deliveryService.getFolderId(deliveryId)
    checkEditDeliveryOnFolder(folderId)
  }

  def checkEditDeliveryOnCreate(folderId: String): Unit = {
    checkEditDeliveryOnFolder(folderId)
  }

  def checkEditDelivery(delivery: Delivery): Unit = {
    checkEditDeliveryOnFolder(delivery.getFolderId)
  }

  def checkEditDeliveryStage(stageId: String): Unit = {
    checkEditStage(stageId, EDIT_RELEASE_DELIVERY)
  }

  def checkEditTrackedItemOnDelivery(deliveryId: String): Unit = {
    val folderId = deliveryService.getFolderId(deliveryId)
    checkEditTrackedItemOnFolder(folderId)
  }

  def checkEditTrackedItemOnStage(stageId: String): Unit = {
    checkEditStage(stageId, EDIT_DELIVERY_TRACKED_ITEM_PERMISSION_SET: _*)
  }

  def checkViewDeliveryPattern(patternId: String): Unit = {
    val folderId = deliveryPatternService.getFolderId(patternId)
    checkViewDeliveryPatternOnFolder(folderId)
  }

  def checkViewDeliveryPattern(pattern: Delivery): Unit = {
    val folderId = pattern.getFolderId
    checkViewDeliveryPatternOnFolder(folderId)
  }

  def checkEditDeliveryPattern(patternId: String): Unit = {
    val folderId = deliveryPatternService.getFolderId(patternId)
    checkEditDeliveryPatternOnFolder(folderId)
  }

  def checkEditDeliveryPattern(pattern: Delivery): Unit = {
    val folderId = pattern.getFolderId
    checkEditDeliveryPatternOnFolder(folderId)
  }

  private def checkViewDeliveryOnFolder(folderId: String): Unit = {
    if (!permissions.hasGlobalPermission(XLReleasePermissions.AUDIT_ALL)) {
      checkPermissionOnFolder(folderId, VIEW_DELIVERY_PERMISSION_SET: _*)
    }
  }

  private def checkEditDeliveryOnFolder(folderId: String): Unit = {
    checkPermissionOnFolder(folderId, EDIT_DELIVERY_PERMISSION_SET: _*)
  }

  private def checkEditTrackedItemOnFolder(folderId: String): Unit = {
    checkPermissionOnFolder(folderId, EDIT_DELIVERY_TRACKED_ITEM_PERMISSION_SET: _*)
  }

  private def checkViewDeliveryPatternOnFolder(folderId: String): Unit = {
    if (!permissions.hasGlobalPermission(XLReleasePermissions.AUDIT_ALL)) {
      checkPermissionOnFolder(folderId, VIEW_DELIVERY_PATTERN_PERMISSION_SET: _*)
    }
  }

  private def checkEditDeliveryPatternOnFolder(folderId: String): Unit = {
    checkPermissionOnFolder(folderId, EDIT_DELIVERY_PATTERN_PERMISSION_SET: _*)
  }

  private def checkEditStage(stageId: String, permissionsToVerify: Permission*): Unit = {
    val deliveryId = deliveryIdFrom(stageId)
    val folderId = deliveryPatternService.getFolderId(deliveryId)
    if (!hasPermissionOnFolder(folderId, permissionsToVerify: _*) && !hasStageOwner(deliveryId, stageId)) {
      throw PermissionDeniedException.withMessage(s"You do not have ${permissionsToVerify.mkString(" or ")} permission or ownership on stage $stageId")
    }
  }

  private def hasStageOwner(deliveryId: String, stageId: String): Boolean = {
    val delivery = deliveryService.getDeliveryOrPattern(deliveryId)
    val stage = delivery.getStageByIdOrTitle(stageId)
    stage.isOwner(getAuthenticatedUserName) || (stage.hasTeam && permissions.isMemberOrRoleOf(delivery.getFolderId, stage.getTeam))
  }

  private def checkPermissionOnFolder(folderId: String, permissionsToVerify: Permission*): Unit = {
    permissions.checkAny(folderId, permissionsToVerify: _*)
  }

  private def hasPermissionOnFolder(folderId: String, permissionsToVerify: Permission*): Boolean = {
    permissionsToVerify.toArray.exists(permissions.hasPermission(_, folderId))
  }

}
