package com.xebialabs.xlrelease.delivery.activity;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.xlrelease.activity.ActivityOps;
import com.xebialabs.xlrelease.delivery.events.DeliveryEvent;
import com.xebialabs.xlrelease.domain.ActivityCategory;
import com.xebialabs.xlrelease.domain.ActivityLogEntry;

import static com.google.common.collect.Sets.newLinkedHashSet;
import static com.xebialabs.xlrelease.domain.ActivityCategory.DELIVERY_EDIT;
import static com.xebialabs.xlrelease.domain.ActivityCategory.IMPORTANT;
import static com.xebialabs.xlrelease.domain.ActivityCategory.LIFECYCLE;

public enum DeliveryActivity implements ActivityOps {
    PATTERN_CREATED("Created pattern '%s'", LIFECYCLE),
    PATTERN_UPDATED("Updated pattern '%s'", DELIVERY_EDIT),
    PATTERN_CREATED_FROM_AS_CODE("Created pattern from as-code %s", LIFECYCLE),
    PATTERN_UPDATED_FROM_AS_CODE("Updated pattern from as-code %s", DELIVERY_EDIT),
    PATTERN_DELETED("Deleted pattern '%s'", LIFECYCLE),

    DELIVERY_CREATED("Created delivery '%s'", LIFECYCLE),
    DELIVERY_UPDATED("Updated delivery '%s'", DELIVERY_EDIT),
    DELIVERY_DELETED("Deleted delivery '%s'", LIFECYCLE),
    DELIVERY_COMPLETED("Completed delivery '%s'", LIFECYCLE, IMPORTANT),

    STAGE_CREATED("Created stage '%s'", DELIVERY_EDIT),
    STAGE_UPDATED("Updated stage '%s'. %s.", DELIVERY_EDIT),
    STAGE_REMOVED("Removed stage '%s'", DELIVERY_EDIT, IMPORTANT),
    STAGE_REOPENED("Reopened stage '%s'", LIFECYCLE),
    STAGE_STARTED("Started stage '%s'", LIFECYCLE, IMPORTANT),
    STAGE_COMPLETED("Completed stage '%s'", LIFECYCLE, IMPORTANT),

    ITEM_CREATED("Created tracked item '%s'", DELIVERY_EDIT),
    ITEM_RENAMED("Renamed tracked item '%s' to '%s'", DELIVERY_EDIT),
    ITEM_REMOVED("Removed tracked item '%s'", DELIVERY_EDIT, IMPORTANT),
    ITEM_DESCOPED("Descoped tracked item '%s'", DELIVERY_EDIT, IMPORTANT),
    ITEM_RESCOPED("Rescoped tracked item '%s'", DELIVERY_EDIT, IMPORTANT),

    ITEM_AVAILABLE("Tracked item '%s' in progress in stage '%s'", LIFECYCLE),
    ITEM_COMPLETED("Completed tracked item '%s' in stage '%s' from release '%s'", LIFECYCLE, IMPORTANT),
    ITEM_COMPLETED_MANUALLY("Completed tracked item '%s' in stage '%s' manually", LIFECYCLE, IMPORTANT),
    ITEM_SKIPPED("Skipped tracked item '%s' in stage '%s'", LIFECYCLE, IMPORTANT),
    ITEM_RESET("Reset tracked item '%s' in stage '%s' to in progress state from release '%s'", LIFECYCLE, IMPORTANT),
    ITEM_RESET_MANUALLY("Reset tracked item '%s' in stage '%s' to in progress state manually", LIFECYCLE, IMPORTANT),
    ITEM_TRANSITION_APPROVED("Approved transition '%s' of tracked item '%s'", LIFECYCLE, IMPORTANT),

    TRANSITION_CREATED("Created transition '%s'", DELIVERY_EDIT),
    TRANSITION_UPDATED("Updated transition '%s'", DELIVERY_EDIT),
    TRANSITION_REMOVED("Removed transition '%s'", DELIVERY_EDIT),
    TRANSITION_EXECUTED_MANUALLY("Executed transition '%s' manually", LIFECYCLE, IMPORTANT),
    TRANSITION_EXECUTED_AUTOMATICALLY("Executed transition '%s' automatically due to condition completion", LIFECYCLE, IMPORTANT),

    CONDITION_SATISFIED("Satisfied %s '%s' on transition '%s'", LIFECYCLE),
    CONDITION_UPDATED("Updated condition '%s' on transition '%s'", DELIVERY_EDIT), // TODO: log changes?

    OTHER("", ActivityCategory.OTHER);

    private String message;
    private Set<ActivityCategory> categories;

    DeliveryActivity(String message, ActivityCategory... categories) {
        this.message = message;
        Set<ActivityCategory> myCategories = newLinkedHashSet();
        if (categories != null) {
            Collections.addAll(myCategories, categories);
        }
        this.categories = Collections.unmodifiableSet(myCategories);
    }

    public ActivityLogEntry create(Date eventTime, String username, Type targetType, String targetId, Object... parameters) {
        return create(eventTime, name(), username, targetType, targetId, message, parameters);
    }

    public ActivityLogEntry create(DeliveryEvent ev, BaseConfigurationItem target, Object... parameters) {
        return create(ev.timestamp(), ev.username(), target.getType(), target.getId(), parameters);
    }

    @Override
    public String getName() {
        return name();
    }

    public Set<ActivityCategory> getCategories() {
        return categories;
    }

    private static final Map<String, DeliveryActivity> BY_NAME;

    static {
        ImmutableMap.Builder<String, DeliveryActivity> builder = ImmutableMap.builder();
        for (DeliveryActivity value : DeliveryActivity.values()) {
            builder.put(value.name(), value);
        }
        BY_NAME = builder.build();
    }

    public static DeliveryActivity safeValueOf(String name) {
        DeliveryActivity activity = BY_NAME.get(name);
        return (activity == null ? OTHER : activity);
    }


}

