import { call, put, select } from 'redux-saga/effects';
import { SagaIterator } from 'redux-saga';

import { DeliveryPatternSidebarProvider, selectDelivery } from './delivery-pattern-sidebar-provider';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { Delivery } from '../../Tasks/js/components/types';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';

const { setSidebarMenu, setBackItem } = navigation.actions;

describe('DeliveryPatternSidebarProvider', () => {
    const deliveryPatternSidebarProvider = new DeliveryPatternSidebarProvider();

    const path = '/delivery-patterns/Deliveries/Deliveryb54aafcfc134439087a2189553d775a7';

    describe('isSupported', () => {
        it('should be true for delivery patterns url', () => {
            expect(deliveryPatternSidebarProvider.isSupported(path)).toBe(true);
        });
    });

    describe('selectDelivery', () => {
        const myState = {
            deliveryPatterns: {
                delivery: 'my delivery',
            },
        };
        it('should return proper path on state', () => {
            expect(selectDelivery(myState)).toBe('my delivery');
        });
    });

    describe('provide', () => {
        const stateParams = {
            folderId: 'Folder1',
        };

        const delivery: Delivery = {
            id: 'Deliveryb54aafcfc134439087a2189553d775a7',
            stages: [],
            title: 'My pattern',
        };

        const uiExtensionsService = {
            getMenuItems: jest.fn(),
        };

        const deliveryPatternItems = [
            {
                icon: 'rocket',
                label: 'Delivery flow',
                pathSuffix: '',
                properties: {},
                uri: 'include/delivery-patterns/ng/flow/delivery-pattern-flow.html',
                weight: 10,
                permitted: true,
            },
            {
                icon: 'configuration-preferences',
                label: 'Properties',
                pathSuffix: 'properties',
                properties: {},
                uri: 'include/delivery-patterns/ng/properties/delivery-pattern-details.html',
                weight: 20,
                permitted: true,
            },
            {
                icon: 'location',
                label: 'Tracked items',
                pathSuffix: 'tracked-items',
                properties: {},
                uri: 'include/delivery-patterns/ng/tracked-items/delivery-pattern-tracked-items.html',
                weight: 30,
                permitted: true,
            },
            {
                icon: 'file-lines',
                label: 'Activity logs',
                pathSuffix: 'logs',
                properties: {},
                uri: 'managed-by-custom-state',
                weight: 40,
                permitted: true,
            },
        ];

        const verifyBegin = (gen: SagaIterator) => {
            expect(gen.next().value).toStrictEqual(call(getAngularService, '$stateParams'));
            expect(gen.next(stateParams).value).toStrictEqual(select(selectDelivery));
            expect(gen.next(delivery).value).toStrictEqual(call(getAngularService, 'UiExtensionsService'));
            expect(gen.next(uiExtensionsService).value).toStrictEqual(
                call([uiExtensionsService, uiExtensionsService.getMenuItems], 'xlrelease.DeliveryPatternPageMenu', delivery),
            );
        };

        const verifyEnd = (gen: SagaIterator) => {
            expect(gen.next(deliveryPatternItems).value).toStrictEqual(
                put(
                    setBackItem({
                        pathSuffix: `folders/${stateParams.folderId}/delivery-patterns`,
                        title: 'patterns',
                        text: 'My pattern',
                    }),
                ),
            );
            expect(gen.next(deliveryPatternItems).done).toBe(true);
        };

        type selectedType = 'Delivery flow' | 'Tracked Items' | 'Properties' | 'Activity logs';
        const createSidebar = (selectedText: selectedType) => {
            return {
                sidebarMenu: [
                    {
                        pathSuffix: 'folders/Folder1/delivery-patterns/Deliveryb54aafcfc134439087a2189553d775a7',
                        selected: selectedText === 'Delivery flow',
                        startIconId: 'rocket',
                        text: 'Delivery flow',
                        weight: 10,
                    },
                    {
                        pathSuffix: 'folders/Folder1/delivery-patterns/Deliveryb54aafcfc134439087a2189553d775a7/properties',
                        selected: selectedText === 'Properties',

                        startIconId: 'configuration-preferences',
                        text: 'Properties',
                        weight: 20,
                    },
                    {
                        pathSuffix: 'folders/Folder1/delivery-patterns/Deliveryb54aafcfc134439087a2189553d775a7/tracked-items',
                        selected: selectedText === 'Tracked Items',
                        startIconId: 'location',
                        text: 'Tracked items',
                        weight: 30,
                    },
                    {
                        pathSuffix: 'folders/Folder1/delivery-patterns/Deliveryb54aafcfc134439087a2189553d775a7/logs',
                        selected: selectedText === 'Activity logs',
                        startIconId: 'file-lines',
                        text: 'Activity logs',
                        weight: 40,
                    },
                ],
                withFolder: false,
            };
        };

        it('should call proper sagas for default page', () => {
            const gen: SagaIterator = deliveryPatternSidebarProvider.provide(path);
            verifyBegin(gen);
            expect(gen.next(deliveryPatternItems).value).toStrictEqual(put(setSidebarMenu(createSidebar('Delivery flow'))));
            verifyEnd(gen);
        });
        it('should call proper sagas for properties page', () => {
            const gen: SagaIterator = deliveryPatternSidebarProvider.provide(`${path}/properties`);
            verifyBegin(gen);
            expect(gen.next(deliveryPatternItems).value).toStrictEqual(put(setSidebarMenu(createSidebar('Properties'))));
            verifyEnd(gen);
        });
        it('should call proper sagas for tracked items page', () => {
            const gen: SagaIterator = deliveryPatternSidebarProvider.provide(`${path}/tracked-items`);
            verifyBegin(gen);
            expect(gen.next(deliveryPatternItems).value).toStrictEqual(put(setSidebarMenu(createSidebar('Tracked Items'))));
            verifyEnd(gen);
        });
        it('should call proper sagas for activity logs', () => {
            const gen: SagaIterator = deliveryPatternSidebarProvider.provide(`${path}/logs`);
            verifyBegin(gen);
            expect(gen.next(deliveryPatternItems).value).toStrictEqual(put(setSidebarMenu(createSidebar('Activity logs'))));
            verifyEnd(gen);
        });
    });
});
