package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.api.v1.DeliveryPatternApi;
import com.xebialabs.xlrelease.api.v1.forms.*;
import com.xebialabs.xlrelease.delivery.actors.DeliveryActorService;
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissionChecker;
import com.xebialabs.xlrelease.delivery.service.DeliveryExecutionService;
import com.xebialabs.xlrelease.delivery.service.DeliveryPatternService;
import com.xebialabs.xlrelease.domain.delivery.Delivery;
import com.xebialabs.xlrelease.domain.delivery.Stage;
import com.xebialabs.xlrelease.domain.delivery.TrackedItem;
import com.xebialabs.xlrelease.domain.delivery.Transition;
import com.xebialabs.xlrelease.exception.LogFriendlyConcurrentModificationException;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Page;

import io.micrometer.core.annotation.Timed;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.domain.utils.DeliveryUtils.deliveryIdFrom;
import static java.util.Objects.isNull;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.springframework.util.StringUtils.hasText;

@Controller
public class DeliveryPatternApiImpl implements DeliveryPatternApi {
    private final DeliveryActorService deliveryActorService;
    private final DeliveryPatternService deliveryPatternService;
    private final DeliveryExecutionService deliveryExecutionService;
    private final DeliveryPermissionChecker permissions;

    @Autowired
    public DeliveryPatternApiImpl(DeliveryActorService deliveryActorService,
                                  DeliveryPatternService deliveryPatternService,
                                  DeliveryExecutionService deliveryExecutionService,
                                  DeliveryPermissionChecker permissions) {
        this.deliveryActorService = deliveryActorService;
        this.deliveryPatternService = deliveryPatternService;
        this.deliveryExecutionService = deliveryExecutionService;
        this.permissions = permissions;
    }

    @Timed
    @Override
    public Delivery createPattern(Delivery pattern) {
        checkNotNull(pattern, "pattern");
        checkNotNull(pattern.getFolderId(), "pattern.folderId");
        permissions.checkEditDeliveryPattern(pattern);
        pattern.getTransitions().forEach(Transition::checkRestrictions);
        return deliveryPatternService.createDeliveryPattern(pattern);
    }

    @Timed
    @Override
    public Delivery getPattern(String patternId) {
        permissions.checkViewDeliveryPattern(patternId);
        return deliveryPatternService.getPattern(patternId);
    }

    @Timed
    @Override
    public Delivery getPatternByIdOrTitle(String patternIdOrTitle) {
        checkArgument(hasText(patternIdOrTitle), "Pattern Id or title is required");

        Delivery pattern = deliveryPatternService.getPatternByIdOrTitle(patternIdOrTitle);
        permissions.checkViewDeliveryPattern(pattern.getId());
        return pattern;
    }

    @Timed
    @Override
    public boolean checkTitleUnique(ValidatePattern validation) {
        return !deliveryPatternService.existsPatternWithTitle(validation.getId(), validation.getTitle());
    }

    @Timed
    @Override
    public Delivery updatePattern(String patternId, Delivery pattern) {
        permissions.checkEditDeliveryPattern(patternId);
        permissions.checkEditDeliveryPattern(pattern);
        pattern.setId(patternId);
        return deliveryActorService.updatePattern(pattern);
    }

    @Timed
    @Override
    public Delivery updatePattern(Delivery pattern) {
        return updatePattern(pattern.getId(), pattern);
    }

    @Timed
    @Override
    public void deletePattern(String patternId) {
        permissions.checkEditDeliveryPattern(patternId);
        deliveryActorService.deletePattern(patternId);
    }

    @Timed
    @Override
    public Delivery duplicatePattern(String patternId, DuplicateDeliveryPattern duplicateDeliveryPattern) {
        permissions.checkEditDeliveryPattern(patternId);
        return deliveryActorService.duplicatePattern(patternId, isNull(duplicateDeliveryPattern) ? new DuplicateDeliveryPattern() : duplicateDeliveryPattern);
    }

    @Timed
    @Override
    public Delivery createDeliveryFromPattern(String patternId, CreateDelivery createDelivery) {
        checkNotNull(createDelivery, "createDelivery");
        checkNotNull(createDelivery.getFolderId(), "createDelivery.folderId");
        permissions.checkEditDeliveryOnCreate(createDelivery.getFolderId());
        return deliveryActorService.createDeliveryFromPattern(patternId, createDelivery);
    }

    @Timed
    @Override
    public List<Delivery> searchPatterns(DeliveryPatternFilters filters,
                                         Long page,
                                         Long resultsPerPage) {
        checkArgument(resultsPerPage <= 100, "Number of results per page cannot be more than 100");
        if (null == filters) {
            filters = new DeliveryPatternFilters();
        }
        return deliveryPatternService.searchPatterns(filters, Page.parse(ofNullable(page), ofNullable(resultsPerPage), empty()), true);
    }

    @Timed
    @Override
    public List<Delivery> searchPatterns(DeliveryPatternFilters filters) {
        return searchPatterns(filters, 0L, 100L);
    }

    // STAGE ENDPOINTS

    @Timed
    @Override
    public Stage createStage(String patternId, Stage stage) {
        return this.createStage(patternId, stage, null);
    }

    @Timed
    @Override
    public Stage createStage(String patternId, Stage stage, Integer position) {
        permissions.checkEditDeliveryPattern(patternId);
        return deliveryActorService.addStage(patternId, stage, ofNullable(position));
    }

    @Timed
    @Override
    public Stage createStage(String patternId, CreateDeliveryStage form) {
        permissions.checkEditDeliveryPattern(patternId);
        return deliveryActorService.addStageBetween(patternId, form.getStage(), ofNullable(form.getBefore()), ofNullable(form.getAfter()));
    }

    @Timed
    @Override
    public List<Stage> getStages(String patternId) {
        permissions.checkViewDeliveryPattern(patternId);
        return deliveryPatternService.getStages(patternId);
    }

    @Timed
    @Override
    public Stage updateStage(Stage stage) {
        return updateStage(stage.getId(), stage);
    }

    @Timed
    @Override
    public Stage updateStage(String stageId, Stage stage) {
        return updateStage(stageId, stage, false);
    }

    @Timed
    @Override
    public Stage updateStageFromBatch(String stageId, Stage stage) {
        return updateStage(stageId, stage, true);
    }

    private Stage updateStage(String stageId, Stage stage, Boolean fromBatch) {
        stage.setId(stageId);
        String patternId = deliveryIdFrom(stageId);
        permissions.checkEditDeliveryPattern(patternId);

        try {
            return deliveryActorService.updateStage(patternId, stage, fromBatch);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    @Timed
    @Override
    public void deleteStage(String stageId) {
        String patternId = deliveryIdFrom(stageId);
        permissions.checkEditDeliveryPattern(patternId);

        try {
            deliveryActorService.deleteStage(patternId, stageId);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    // TRANSITION ENDPOINTS

    @Timed
    @Override
    public Transition createTransition(String stageId, Transition transition) {
        checkNotNull(transition, "Transition");
        String patternId = deliveryIdFrom(stageId);
        permissions.checkEditDeliveryPattern(patternId);
        transition.checkRestrictions();

        try {
            return deliveryActorService.addTransition(patternId, stageId, transition);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    @Timed
    @Override
    public Transition updateTransition(String transitionId, Transition transition) {
        checkNotNull(transition, "Transition");
        String patternId = deliveryIdFrom(transitionId);
        permissions.checkEditDeliveryPattern(patternId);

        transition.setId(transitionId);
        transition.checkRestrictions();
        try {
            return deliveryActorService.updateTransition(patternId, transition);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    @Timed
    @Override
    public Transition updateTransition(Transition transition) {
        return updateTransition(transition.getId(), transition);
    }

    @Timed
    @Override
    public void deleteTransition(String transitionId) {
        String patternId = deliveryIdFrom(transitionId);
        permissions.checkEditDeliveryPattern(patternId);

        try {
            deliveryActorService.deleteTransition(patternId, transitionId);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    // TRACKED ITEM ENDPOINTS

    @Timed
    @Override
    public TrackedItem createTrackedItem(String patternId, TrackedItem item) {
        permissions.checkEditDeliveryPattern(patternId);
        return deliveryActorService.createTrackedItem(patternId, item);
    }

    @Timed
    @Override
    public List<TrackedItem> getTrackedItems(String patternId) {
        permissions.checkViewDeliveryPattern(patternId);
        return deliveryExecutionService.getTrackedItems(patternId);
    }

    @Timed
    @Override
    public TrackedItem updateTrackedItem(String itemId, TrackedItem item) {
        item.setId(itemId);
        String patternId = deliveryIdFrom(itemId);
        permissions.checkEditDeliveryPattern(patternId);
        try {
            return deliveryActorService.updateTrackedItem(patternId, item);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    @Timed
    @Override
    public TrackedItem updateTrackedItem(TrackedItem item) {
        return updateTrackedItem(item.getId(), item);
    }

    @Timed
    @Override
    public void deleteTrackedItem(String itemId) {
        String patternId = deliveryIdFrom(itemId);
        permissions.checkEditDeliveryPattern(patternId);
        try {
            deliveryActorService.deleteTrackedItem(patternId, itemId);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }
}
