package com.xebialabs.xlrelease.delivery.builder

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.delivery.{Delivery, Stage}
import com.xebialabs.xlrelease.domain.folder.Folder

import java.time.Instant
import java.util.Date
import java.util.UUID.randomUUID
import scala.annotation.varargs


object DeliveryBuilder {
  def delivery(title: String): DeliveryContext = DeliveryContext(title)
}

case class DeliveryContext(title: String, folder: Option[Folder], startDate: Option[Date], endDate: Option[Date], stages: List[Stage]) {

  @varargs
  def withStages(stage: Stage*): DeliveryContext = this.copy(title, folder, startDate, endDate, stage.toList)

  def inFolder(folder: Folder): DeliveryContext = this.copy(title, Some(folder), startDate, endDate, stages)

  def withStartDate(startDate: Date): DeliveryContext = this.copy(title, folder, Some(startDate), endDate, stages)

  def withStartDate(startDate: Instant): DeliveryContext = this.copy(title, folder, Some(Date.from(startDate)), endDate, stages)

  def withEndDate(endDate: Date): DeliveryContext = this.copy(title, folder, startDate, Some(endDate), stages)

  def withEndDate(endDate: Instant): DeliveryContext = this.copy(title, folder, startDate, Some(Date.from(endDate)), stages)

  def build: Delivery = {
    val delivery = new Delivery()
    delivery.setId(generateId(Delivery.DELIVERY_ROOT, classOf[Delivery]))

    delivery.setFolderId(folder.getOrElse(throw new IllegalStateException("folder not provided")).getId)

    delivery.setTitle(title)
    delivery.setStartDate(startDate.getOrElse(throw new IllegalStateException("startDate not provided")))
    delivery.setEndDate(endDate.getOrElse(throw new IllegalStateException("endDate not provided")))
    delivery.updateDuration()

    stages.foreach { stage =>
      stage.setId(generateId(delivery.getId, classOf[Stage]))
      delivery.addStage(stage)
    }
    delivery
  }

  private def generateId(parentId: String, clazz: Class[_]): String =
    s"$parentId/${Type.valueOf(clazz).getName}${randomUUID().toString}"

}

object DeliveryContext {
  def apply(title: String) = new DeliveryContext(title, None, None, None, List.empty)
}


object StageBuilder {
  def stage(title: String): StageContext = StageContext(title)
}

class StageContext(title: String) {
  def build: Stage = {
    new Stage(title)
  }
}

object StageContext {
  def apply(title: String): StageContext = new StageContext(title)
}


