import { SagaIterator } from 'redux-saga';
import { call, put } from 'redux-saga/effects';
import { FolderDeliveriesSidebarProvider } from './folder-deliveries-sidebar-provider';
import { FolderPathService, MainNavigationItem } from '../../../../../../../../../core/xlr-ui/app/features/tasks/types/angular';
import { getFolderOrWaitForIt } from '../../../../../../../../../core/xlr-ui/app/features/folders/ducks/folders.saga';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';

const { setSidebarMenu, setBackItem, updateBreadcrumbs } = navigation.actions;
describe('FolderDeliveriesSidebarProvider', () => {
    const folderDeliveriesSidebarProvider = new FolderDeliveriesSidebarProvider();

    describe('isSupported', () => {
        it('should be true for delivery patterns url', () => {
            expect(folderDeliveriesSidebarProvider.isSupported('/folders/Applications/FolderSamplesAndTutorials/delivery-patterns')).toBe(true);
        });
        it('should be true for deliveries url', () => {
            expect(folderDeliveriesSidebarProvider.isSupported('/folders/Applications/FolderSamplesAndTutorials/deliveries')).toBe(true);
        });

        it('should be false when we are not in a folder', () => {
            expect(folderDeliveriesSidebarProvider.isSupported('/deliveries')).toBe(false);
        });

        it('should be false when opening a delivery pattern details', () => {
            expect(
                folderDeliveriesSidebarProvider.isSupported('/folders/Applications/FolderSamplesAndTutorials/delivery-patterns/Deliveries/Delivery123'),
            ).toBe(false);
        });

        it('should be false when opening a delivery details', () => {
            expect(folderDeliveriesSidebarProvider.isSupported('/folders/Applications/FolderSamplesAndTutorials/deliveries/Deliveries/Delivery123')).toBe(
                false,
            );
        });
    });

    describe('provide', () => {
        const UiExtensionsService = {
            getMenuItems: jest.fn(),
        };
        const folderPathService: FolderPathService = {
            getPathFromParentFolderId: () => [
                {
                    id: 'Applications',
                    title: 'My Root',
                },
            ],
            getPathFromReleaseId: jest.fn(),
        };

        const selectedFolder = {
            id: 'Applications/FolderSamplesAndTutorials',
            title: 'My Folder',
        };

        const folderTabs: MainNavigationItem[] = [
            {
                label: 'Deliveries',
                id: 'xlrelease.sidebar.Folder.Deliveries',
                permitted: true,
                items: [
                    {
                        icon: 'pattern',
                        label: 'Patterns',
                        pathSuffix: 'patterns',
                        weight: 10,
                        permitted: true,
                    },
                    {
                        icon: 'pattern',
                        label: 'Deliveries overview',
                        pathSuffix: 'deliveries',
                        weight: 20,
                        permitted: true,
                    },
                ],
            },
        ];

        it('should call proper sagas', () => {
            const saga: SagaIterator = folderDeliveriesSidebarProvider.provide('/folders/Applications/FolderSamplesAndTutorials/deliveries');
            expect(saga.next().value).toMatchObject(call(getFolderOrWaitForIt));
            expect(saga.next(selectedFolder).value).toMatchObject(call(getAngularService, 'UiExtensionsService'));
            expect(saga.next(UiExtensionsService).value).toMatchObject(
                call([UiExtensionsService, UiExtensionsService.getMenuItems], 'xlrelease.sidebar.Folder', selectedFolder),
            );
            expect(saga.next(folderTabs).value).toMatchObject(
                put(
                    setSidebarMenu({
                        sidebarMenu: [
                            {
                                pathSuffix: 'folders/Applications/FolderSamplesAndTutorials/patterns',
                                selected: false,
                                startIconId: 'pattern',
                                text: 'Patterns',
                            },
                            {
                                pathSuffix: 'folders/Applications/FolderSamplesAndTutorials/deliveries',
                                selected: true,
                                startIconId: 'pattern',
                                text: 'Deliveries overview',
                            },
                        ],
                        withFolder: false,
                    }),
                ),
            );
            expect(saga.next().value).toMatchObject(
                put(
                    setBackItem({
                        pathSuffix: `folders/${selectedFolder.id}/templates`,
                        title: selectedFolder.title,
                        text: selectedFolder.title,
                    }),
                ),
            );
            expect(saga.next().value).toMatchObject(call(getAngularService, 'FolderPathService'));
            expect(saga.next(folderPathService).value).toMatchObject(call([folderPathService, folderPathService.getPathFromParentFolderId], selectedFolder.id));
            expect(saga.next(folderPathService.getPathFromParentFolderId('')).value).toMatchObject(
                put(
                    updateBreadcrumbs([
                        {
                            href: '/folders/Applications/deliveries',
                            text: 'My Root',
                        },
                    ]),
                ),
            );
            expect(saga.next().done).toBe(true);
        });
    });
});
