import React, { useEffect, useState } from 'react';
import { DotAutoComplete, AutoCompleteOption } from '@digital-ai/dot-components';
import IdsFactory from '../../../../../../../../../../../core/xlr-ui/app/js/util/ids';
import { TaskInputCommonProps } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/config/components/types';
import { StringWithVariables } from './string-with-variables.component';
import { useAppDispatch, useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { getDeliveryTasksState, deliveryTasks } from '../ducks/delivery-tasks.reducer';
import { Delivery } from '../types';
import { isDeliveryId } from '../helper';
import { getPropertyFromInputProperties } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/helper/task.helper';
import './delivery-or-pattern.component.less';
import { isVariable } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/helper/variables.helper';

const Ids = IdsFactory();
const { loadDeliveryOrPattern, searchDeliveriesOrPatterns } = deliveryTasks.actions;

interface AutoCompleteOptionWithId extends AutoCompleteOption {
    id: string;
}

interface DeliveryOrPatternSelectorProps extends TaskInputCommonProps {
    isDelivery: boolean;
}

export const PatternWithVariables = (props: TaskInputCommonProps) => {
    return (
        <StringWithVariables {...props}>
            <DeliveryOrPatternAutocomplete {...props} isDelivery={false} />
        </StringWithVariables>
    );
};

export const DeliveryWithVariables = (props: TaskInputCommonProps) => {
    return (
        <StringWithVariables {...props}>
            <DeliveryOrPatternAutocomplete {...props} isDelivery={true} />
        </StringWithVariables>
    );
};

const findByIdOrName = (deliveries: Array<Delivery>, idOrName: string) => {
    return deliveries.find((d: Delivery) => d.id === idOrName || Ids.getName(d.id) === idOrName || d.title === idOrName) || null;
};

const DeliveryOrPatternAutocomplete = ({ disabled, onChange, property, task, isDelivery }: DeliveryOrPatternSelectorProps) => {
    const dispatch = useAppDispatch();
    const { deliveryOrPattern, deliveriesOrPatterns } = useAppSelector(getDeliveryTasksState);
    const [selectedDelivery, setSelectedDelivery] = useState<AutoCompleteOptionWithId | null>(null);
    const [hasPatternViewPermission, setHasPatternViewPermission] = useState(true);
    const [isDeliveryFound, setDeliveryFound] = useState(true);
    const [description, setDescription] = useState(property.description);
    const patternId = getPropertyFromInputProperties(task, property.name) as string;

    useEffect(() => {
        dispatch(searchDeliveriesOrPatterns({ isDelivery }));
    }, [property.fqn]);

    useEffect(() => {
        if (patternId && deliveriesOrPatterns.length > 0 && isDeliveryId(patternId)) {
            dispatch(loadDeliveryOrPattern({ isDelivery, id: patternId }));
        } else if (patternId && deliveriesOrPatterns.length > 0 && !isVariable(patternId)) {
            const id = findByIdOrName(deliveriesOrPatterns, patternId)?.id;
            id && dispatch(loadDeliveryOrPattern({ isDelivery, id }));
        } else if (patternId && deliveriesOrPatterns.length === 0) {
            dispatch(loadDeliveryOrPattern({ isDelivery, id: patternId }));
        } else {
            setSelectedDelivery(null);
        }
    }, [patternId, deliveriesOrPatterns]);

    useEffect(() => {
        if (!deliveryOrPattern) return;
        setDeliveryFound(true);
        if (isDelivery) {
            if (deliveryOrPattern.id === 'NOT_FOUND') {
                setDeliveryFound(false);
                setSelectedDelivery({ id: patternId, title: patternId });
                setDescription('Currently selected delivery was deleted');
            } else {
                setHasPatternViewPermission(true);
                const foundDeliveryOnOptions = findByIdOrName(deliveriesOrPatterns, patternId);
                setSelectedDelivery(foundDeliveryOnOptions);
            }
        } else {
            if (deliveryOrPattern.id === 'forbiddenFolderNotNull' && deliveryOrPattern.title === '') {
                setHasPatternViewPermission(false);
                setDescription('You do not have permissions to view the selected pattern');
                const pId = Ids.getName(Ids.toDomainId(patternId));
                const patternWithId: AutoCompleteOptionWithId = { id: patternId, title: pId };
                setSelectedDelivery(patternWithId);
            } else if (deliveryOrPattern.id === null) {
                setDeliveryFound(false);
                setSelectedDelivery({ id: patternId, title: patternId });
                setDescription('Currently selected pattern was deleted');
            } else {
                setHasPatternViewPermission(true);
                const foundDeliveryOnOptions = findByIdOrName(deliveriesOrPatterns, patternId);
                setSelectedDelivery(foundDeliveryOnOptions);
            }
        }
    }, [deliveryOrPattern]);

    const handleDeliveryChange = (value?: AutoCompleteOptionWithId) => {
        onChange(value ? value.id : null);
    };

    return (
        <div className="delivery-or-pattern">
            <DotAutoComplete
                error={!hasPatternViewPermission || !isDeliveryFound}
                freesolo={false}
                helperText={description}
                inputId={property.name}
                key={selectedDelivery?.id}
                label={property.label}
                multiple={false}
                onChange={(_e, v) => handleDeliveryChange(v as AutoCompleteOptionWithId)}
                options={deliveriesOrPatterns}
                persistentLabel={true}
                readOnly={disabled}
                required={property.required}
                value={selectedDelivery}
            />
        </div>
    );
};
