import React, { ReactElement } from 'react';
import { ReactWrapper } from 'enzyme';
import { DotAutoComplete } from '@digital-ai/dot-components';
import { mountWithStoreAndTheme } from '../../../../../../../../../../../core/xlr-ui/tests/unit/testing-utils';
import { stringPropertyMock } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/property.mock';
import { taskMock } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/__mocks__/task.mock';
import { PropertiesEnum, TaskInputCommonProps } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/config/components/types';
import { RootState } from '../../../../../../../../../../../core/xlr-ui/app/js/store.types';
import { DeliveryWithVariables, PatternWithVariables } from './delivery-or-pattern.component';

describe('delivery-or-pattern.component', () => {
    const task = {
        ...taskMock,
        inputProperties: {
            string: 'PatternId1',
        },
    };

    const deliveriesOrPatterns = [
        {
            id: 'PatternId1',
            title: 'Pattern 1',
        },
        {
            id: 'PatternId2',
            title: 'Pattern 2',
        },
    ];

    const defaultState = {
        taskDrawer: {
            variables: [{ key: 'var' }],
        },
        deliveryTasks: {
            deliveriesOrPatterns,
            deliveryOrPattern: {
                id: 'PatternId1',
                title: 'Pattern 1',
            },
        },
    };

    const property = {
        ...stringPropertyMock,
    };

    const onChange = jest.fn();
    const dispatch = jest.fn();
    let wrapper: ReactWrapper;

    const mount = (component: ReactElement, state: RootState = defaultState) => {
        wrapper = mountWithStoreAndTheme(component, dispatch, state);
    };

    const defaultProps: TaskInputCommonProps = {
        disabled: false,
        expanded: false,
        onChange,
        propertiesEnum: PropertiesEnum.InputProperties,
        property,
        task,
    };

    const mountPatternWithVariables = (props: TaskInputCommonProps = defaultProps, state: RootState = defaultState) =>
        mount(<PatternWithVariables {...props} />, state);
    const mountDeliveryWithVariables = (props: TaskInputCommonProps = defaultProps, state: RootState = defaultState) =>
        mount(<DeliveryWithVariables {...props} />, state);

    const getDotAutoComplete = () => wrapper.find(DotAutoComplete);

    describe('PatternWithVariables', () => {
        beforeEach(() => {
            dispatch.mockReset();
            onChange.mockReset();
        });

        it('should trigger the search', () => {
            mountPatternWithVariables();
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: false },
                type: 'deliveryTasks/searchDeliveriesOrPatterns',
            });
        });

        it('should trigger loadDeliveryOrPattern when pattern id is defined as id', () => {
            mountPatternWithVariables();
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: false, id: 'PatternId1' },
                type: 'deliveryTasks/loadDeliveryOrPattern',
            });
        });

        it('should trigger loadDeliveryOrPattern when pattern id is defined as title', () => {
            const taskAsTitle = {
                ...taskMock,
                inputProperties: {
                    string: 'Pattern 2',
                },
            };
            mountPatternWithVariables({ ...defaultProps, task: taskAsTitle });
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: false, id: 'PatternId2' },
                type: 'deliveryTasks/loadDeliveryOrPattern',
            });
        });

        it('should trigger loadDeliveryOrPattern when pattern id but there are no patterns loaded', () => {
            const state = {
                ...defaultState,
                deliveryTasks: {
                    ...defaultState.deliveryTasks,
                    deliveriesOrPatterns: [],
                },
            };
            const taskWithUnknownPattern = {
                ...taskMock,
                inputProperties: {
                    string: 'Pattern123123123',
                },
            };
            mountPatternWithVariables({ ...defaultProps, task: taskWithUnknownPattern }, state);
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: false, id: 'Pattern123123123' },
                type: 'deliveryTasks/loadDeliveryOrPattern',
            });
        });

        it('should pass proper props when a pattern id is defined', () => {
            mountPatternWithVariables();
            const autocompleteProps = getDotAutoComplete().props();
            expect(autocompleteProps.value).toStrictEqual({
                id: 'PatternId1',
                title: 'Pattern 1',
            });
            expect(autocompleteProps.readOnly).toBe(false);
            expect(autocompleteProps.error).toBe(false);
            expect(autocompleteProps.freesolo).toBe(false);
            expect(autocompleteProps.helperText).toBe(property.description);
            expect(autocompleteProps.inputId).toBe(property.name);
            expect(autocompleteProps.label).toBe(property.label);
            expect(autocompleteProps.multiple).toBe(false);
            expect(autocompleteProps.persistentLabel).toBe(true);
            expect(autocompleteProps.options).toStrictEqual(deliveriesOrPatterns);
            expect(autocompleteProps.required).toBe(property.required);
        });

        it('should update error and description when folder is forbidden', () => {
            const stateForbidden = {
                ...defaultState,
                deliveryTasks: {
                    ...defaultState.deliveryTasks,
                    deliveryOrPattern: {
                        id: 'forbiddenFolderNotNull',
                        title: '',
                    },
                },
            };

            mountPatternWithVariables(defaultProps, stateForbidden);
            const autocompleteProps = getDotAutoComplete().props();
            expect(autocompleteProps.error).toBe(true);
            expect(autocompleteProps.helperText).toBe('You do not have permissions to view the selected pattern');
        });
        it('should update error and description when pattern is not found', () => {
            const stateNotFound = {
                ...defaultState,
                deliveryTasks: {
                    ...defaultState.deliveryTasks,
                    deliveryOrPattern: {
                        id: null,
                        title: '',
                    },
                },
            };

            mountPatternWithVariables(defaultProps, stateNotFound);
            const autocompleteProps = getDotAutoComplete().props();
            expect(autocompleteProps.error).toBe(true);
            expect(autocompleteProps.helperText).toBe('Currently selected pattern was deleted');
        });
    });

    describe('DeliveryWithVariables', () => {
        beforeEach(() => {
            dispatch.mockReset();
            onChange.mockReset();
        });

        it('should trigger the search', () => {
            mountDeliveryWithVariables();
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: true },
                type: 'deliveryTasks/searchDeliveriesOrPatterns',
            });
        });

        it('should trigger loadDeliveryOrPattern when pattern id is defined and not a variable', () => {
            mountDeliveryWithVariables();
            expect(dispatch).toHaveBeenCalledWith({
                payload: { isDelivery: true, id: 'PatternId1' },
                type: 'deliveryTasks/loadDeliveryOrPattern',
            });
        });

        it('should pass proper props when a pattern id is defined', () => {
            mountDeliveryWithVariables();
            const autocompleteProps = getDotAutoComplete().props();
            expect(autocompleteProps.value).toStrictEqual({
                id: 'PatternId1',
                title: 'Pattern 1',
            });
            expect(autocompleteProps.readOnly).toBe(false);
            expect(autocompleteProps.error).toBe(false);
            expect(autocompleteProps.freesolo).toBe(false);
            expect(autocompleteProps.helperText).toBe(property.description);
            expect(autocompleteProps.inputId).toBe(property.name);
            expect(autocompleteProps.label).toBe(property.label);
            expect(autocompleteProps.multiple).toBe(false);
            expect(autocompleteProps.persistentLabel).toBe(true);
            expect(autocompleteProps.options).toStrictEqual(deliveriesOrPatterns);
            expect(autocompleteProps.required).toBe(property.required);
        });

        it('should update error and description when delivery is not found', () => {
            const stateNotFound = {
                ...defaultState,
                deliveryTasks: {
                    ...defaultState.deliveryTasks,
                    deliveryOrPattern: {
                        id: 'NOT_FOUND',
                        title: '',
                    },
                },
            };

            mountDeliveryWithVariables(defaultProps, stateNotFound);
            const autocompleteProps = getDotAutoComplete().props();
            expect(autocompleteProps.error).toBe(true);
            expect(autocompleteProps.helperText).toBe('Currently selected delivery was deleted');
        });
    });
});
