package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.delivery.security.DeliveryPermissionChecker;
import com.xebialabs.xlrelease.delivery.service.DeliveryActivityLogsService;
import com.xebialabs.xlrelease.repository.ActivityLogRepository;
import com.xebialabs.xlrelease.views.ActivityLogEntryView;
import com.xebialabs.xlrelease.views.LogsFilters;
import com.xebialabs.xlrelease.views.converters.UserViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.api.ApiService.DEFAULT_RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.delivery.activity.DeliveryActivity.ITEM_TRANSITION_APPROVED;

@Path("/activity")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class DeliveryActivityLogsResource extends BaseActivityLogsResource {

    private DeliveryPermissionChecker permissions;
    private final ActivityLogRepository activityLogRepository;

    @Autowired
    public DeliveryActivityLogsResource(final DeliveryActivityLogsService deliveryActivityLogsService,
                                        final UserViewConverter userViewConverter,
                                        final DeliveryPermissionChecker permissions,
                                        final ActivityLogRepository activityLogRepository) {
        super(deliveryActivityLogsService, userViewConverter);
        this.permissions = permissions;
        this.activityLogRepository = activityLogRepository;
    }

    /**
     * Searches in the activity logs of a delivery.
     *
     * @param deliveryId the identifier of the delivery
     * @param filters    the search criteria to use
     * @return the activity logs
     */
    @POST
    @Path("/{deliveryId:.*Delivery[^/-]*}/logs/search")
    public List<ActivityLogEntryView> searchDeliveryLogs(
            @PathParam("deliveryId") String deliveryId,
            @DefaultValue("0") @QueryParam(PAGE) int page,
            @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) int resultsPerPage,
            LogsFilters filters
    ) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE);
        permissions.checkView(deliveryId);

        // sort feature of Pageable is not used, because there is sort order in LogsFilters
        PageRequest pageable = PageRequest.of(page, resultsPerPage, Sort.unsorted());
        return searchLogs(deliveryId, filters, pageable);
    }

    @GET
    @Path("/{deliveryId:.*Delivery[^/-]*}/approvals")
    public List<ActivityLogEntryView> searchTrackedItemApprovals(@PathParam("deliveryId") String deliveryId) {
        permissions.checkViewDelivery(deliveryId);
        return activityLogRepository.findAllLogsOf(deliveryId, ITEM_TRANSITION_APPROVED, Pageable.unpaged()).stream()
                .map(super::toActivityLogEntryView).collect(Collectors.toList());
    }
}
