package com.xebialabs.xlrelease.delivery.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySchema.{RELEASE_DELIVERIES, RELEASE_DELIVERY_MEMBERS}
import com.xebialabs.xlrelease.domain.delivery.Delivery.DELIVERY_ROOT
import com.xebialabs.xlrelease.domain.delivery.DeliveryStatus
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.{CiId, RichCiId}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, RELEASES}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, PersistenceSupport}
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import scala.jdk.CollectionConverters._
import scala.util.Try

@Repository
@IsTransactional
class DeliveryMemberPersistence @Autowired()(implicit @Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                                 @Qualifier("xlrRepositorySqlDialect") val dialect: Dialect,
                                                 deliveryPersistence: DeliveryPersistence,
                                                 folderPersistence: FolderPersistence) extends PersistenceSupport {

  private val STMT_FIND_DELIVERIES_REFERENCING_MEMBER_ID: String =
    s"""SELECT DISTINCT del.${RELEASE_DELIVERIES.ID}
       |FROM ${RELEASE_DELIVERY_MEMBERS.TABLE} delm
       |INNER JOIN ${RELEASE_DELIVERIES.TABLE} del ON del.${RELEASE_DELIVERIES.CI_UID} = delm.${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID}
       |WHERE delm.${RELEASE_DELIVERY_MEMBERS.RELEASE_ID} = :releaseId""".stripMargin

  def findDeliveriesReferencingReleaseId(releaseId: CiId): Seq[CiId] = {
    sqlQuery(STMT_FIND_DELIVERIES_REFERENCING_MEMBER_ID, params("releaseId" -> getName(releaseId.normalized)), _.getString(1)).map(name => s"$DELIVERY_ROOT/$name").toSeq
  }

  private val STMT_FIND_ACTIVE_DELIVERIES_BY_FOLDER_ID: String =
    s"""SELECT DISTINCT del.${RELEASE_DELIVERIES.ID}, del.${RELEASE_DELIVERIES.TITLE}
       |FROM ${RELEASE_DELIVERY_MEMBERS.TABLE} delm
       |INNER JOIN ${RELEASE_DELIVERIES.TABLE} del ON del.${RELEASE_DELIVERIES.CI_UID} = delm.${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID}
       |INNER JOIN ${RELEASES.TABLE} r ON r.${RELEASES.RELEASE_ID} = delm.${RELEASE_DELIVERY_MEMBERS.RELEASE_ID}
       |INNER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |WHERE f.${FOLDERS.CI_UID} IN (:folderUids)
       |AND del.${RELEASE_DELIVERIES.FOLDER_UID} NOT IN (:folderUids)
       |AND del.${RELEASE_DELIVERIES.STATUS} NOT IN (:inactiveStatuses)""".stripMargin

  def findActiveDeliveriesReferencingFolderId(folderId: CiId): Seq[CiIdWithTitle] = {
    val folderSubtree = folderPersistence.findDescendantsById(folderId)
    sqlQuery(STMT_FIND_ACTIVE_DELIVERIES_BY_FOLDER_ID,
      params(
        "folderUids" -> folderSubtree.map(_.uid).asJava,
        "inactiveStatuses" -> DeliveryStatus.INACTIVE_STATUSES.map(_.value()).toSeq.asJava
      ), rs => CiIdWithTitle(s"$DELIVERY_ROOT/${rs.getString(1)}", rs.getString(2)))
      .toSeq
  }

  private val STMT_FIND_MEMBERS_BY_DELIVERY_UIDS =
    s"""SELECT delm.${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID}, f.${FOLDERS.FOLDER_PATH}, f.${FOLDERS.FOLDER_ID}, delm.${RELEASE_DELIVERY_MEMBERS.RELEASE_ID}
       |FROM ${RELEASE_DELIVERY_MEMBERS.TABLE} delm
       |INNER JOIN ${RELEASES.TABLE} r ON r.${RELEASES.RELEASE_ID} = delm.${RELEASE_DELIVERY_MEMBERS.RELEASE_ID}
       |LEFT OUTER JOIN ${FOLDERS.TABLE} f ON f.${FOLDERS.CI_UID} = r.${RELEASES.FOLDER_CI_UID}
       |WHERE delm.${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID} IN (:deliveryUids)""".stripMargin

  def findMembersByDeliveryUid(deliveryUid: CiUid): Seq[String] = findMembersByDeliveryUids(Seq(deliveryUid)).getOrElse(deliveryUid, Seq.empty)

  def findMembersByDeliveryUids(deliveryUids: Seq[Int]): Map[Int, Seq[String]] = if (deliveryUids.nonEmpty) {
    sqlQuery(STMT_FIND_MEMBERS_BY_DELIVERY_UIDS, params("deliveryUids" -> deliveryUids.asJava), rs => {
      val deliveryUid = rs.getInt(RELEASE_DELIVERY_MEMBERS.DELIVERY_UID)
      val folderPath = Option(rs.getString(FOLDERS.FOLDER_PATH)).map(FolderId.apply).getOrElse(FolderId.Root)
      val releaseId = folderPath / rs.getString(FOLDERS.FOLDER_ID) / rs.getString(RELEASE_DELIVERY_MEMBERS.RELEASE_ID)
      (deliveryUid, releaseId.absolute)
    }).groupBy(_._1).view.mapValues(_.map(_._2).toSeq).toMap
  } else {
    Map.empty
  }

  private val STMT_INSERT_MEMBERS: String =
    s"""
       |INSERT INTO ${RELEASE_DELIVERY_MEMBERS.TABLE} (${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID}, ${RELEASE_DELIVERY_MEMBERS.RELEASE_ID})
       |SELECT :deliveryUid, r.${RELEASES.RELEASE_ID}
       |FROM ${RELEASES.TABLE} r
       |WHERE r.${RELEASES.RELEASE_ID} = :releaseId""".stripMargin // on purpose like this so we do not re-insert archived releases

  def insertMembers(deliveryUid: CiUid, releaseIds: Set[CiId]): Unit = if (releaseIds.nonEmpty) {
    sqlBatch(STMT_INSERT_MEMBERS, releaseIds.map { id =>
      Map("deliveryUid" -> deliveryUid, "releaseId" -> getName(id.normalized))
    })
  }

  private val STMT_DELETE_DELIVERY_MEMBERS: String =
    s"""
       |DELETE FROM ${RELEASE_DELIVERY_MEMBERS.TABLE}
       | WHERE
       |  ${RELEASE_DELIVERY_MEMBERS.DELIVERY_UID} = :deliveryUid
       |  AND ${RELEASE_DELIVERY_MEMBERS.RELEASE_ID} = :releaseId
       |""".stripMargin

  def deleteMembers(deliveryUid: CiUid, releaseIds: Set[CiId]): Unit = if (releaseIds.nonEmpty) {
    sqlBatch(STMT_DELETE_DELIVERY_MEMBERS, releaseIds.map(id => Map("deliveryUid" -> deliveryUid, "releaseId" -> getName(id.normalized))))
  }

  private val STMT_DELETE_DELIVERY_MEMBERS_BY_RELEASE_ID: String =
    s"""
       |DELETE FROM ${RELEASE_DELIVERY_MEMBERS.TABLE}
       | WHERE
       |  ${RELEASE_DELIVERY_MEMBERS.RELEASE_ID} = :releaseId
       |""".stripMargin

  def deleteMembersByReleaseId(releaseId: CiId): Unit =
    sqlExec(STMT_DELETE_DELIVERY_MEMBERS_BY_RELEASE_ID, Map("releaseId" -> getName(releaseId.normalized)), ps => Try(ps.execute()))

}
