package com.xebialabs.xlrelease.delivery.transition.condition

import com.xebialabs.xlrelease.delivery.events._
import com.xebialabs.xlrelease.domain.delivery.{Condition, Delivery, Stage, Transition}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@EventListener
class DeliveryConditionHandlers @Autowired()(conditionReactions: java.util.List[DeliveryConditionReaction]) {

  //noinspection ScalaStyle
  @Subscribe
  def onDeliveryEvent(event: DeliveryEvent): Unit = if (shouldProcess(event)) {
    event match {
      case DeliveryCreatedEvent(delivery) => process(created = collectConditions(delivery))
      case DeliveryDeletedEvent(delivery) => process(deleted = collectConditions(delivery))
      case StageCompletedEvent(stage, delivery) => process(deleted = collectConditions(delivery, stage))
      case TransitionUpdatedEvent(original, updated, delivery) if original.isAutomated != updated.isAutomated =>
        process(updated = collectConditions(delivery, delivery.getStageByTransition(updated)))
      case ConditionSatisfiedEvent(condition, transition, delivery) if condition.isLeaf =>
        process(deleted = Seq(ConditionChange(delivery, transition.getStage, transition, condition)))
      case ConditionUpdatedEvent(condition, transition, delivery) =>
        process(updated = collectConditions(delivery, transition.getStage, transition, condition))
      case _ => // noop
    }
  }

  protected def shouldProcess(event: DeliveryEvent): Boolean = !event.delivery.isTemplate

  protected def process(created: Seq[ConditionChange] = Seq.empty,
                        updated: Seq[ConditionChange] = Seq.empty,
                        deleted: Seq[ConditionChange] = Seq.empty): Unit = {
    created.foreach(condition => getReaction(condition).foreach(_.onCreate(condition)))
    updated.foreach(condition => getReaction(condition).foreach(_.onUpdate(condition)))
    deleted.foreach(condition => getReaction(condition).foreach(_.onDelete(condition)))
  }

  protected def getReaction(conditionContext: ConditionChange): Option[DeliveryConditionReaction] = {
    val allReactions = conditionReactions.asScala

    val conditionType = conditionContext.condition.getType
    val allConditionTypes = conditionType +: conditionType.getDescriptor.getSuperClasses.asScala
    val reaction = allConditionTypes
      .map(typ => allReactions.find(_.forType == typ))
      .find(_.isDefined).flatten
    reaction
  }

  protected def collectConditions(delivery: Delivery): Seq[ConditionChange] =
    delivery.getStages.asScala
      .flatMap(stage => collectConditions(delivery, stage)).toSeq

  protected def collectConditions(delivery: Delivery, stage: Stage): Seq[ConditionChange] = {
    Option(stage.getTransition)
      .map(transition => transition.getConditions.asScala.flatMap(collectConditions(delivery, stage, transition, _)).toSeq)
      .getOrElse(Seq.empty)
  }

  protected def collectConditions(delivery: Delivery, stage: Stage, transition: Transition, condition: Condition): Seq[ConditionChange] = {
    condition.getLeafConditions.asScala.map(ConditionChange(delivery, stage, transition, _)).toSeq
  }
}
