package com.xebialabs.xlrelease.delivery.service

import com.codahale.metrics.annotation.Timed
import com.google.common.annotations.VisibleForTesting
import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.xlrelease.api.v1.forms.{AbstractDeliveryFilters, DeliveryOrderMode}
import com.xebialabs.xlrelease.api.v1.views.{DeliveryFlowReleaseInfo, DeliveryTimeline}
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.delivery.events.{ConditionUpdatedEvent, DeliveryCreatedEvent, StageUpdatedEvent, TransitionUpdatedEvent}
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.CiIdWithTitle
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissions.VIEW_DELIVERY_PERMISSION_SET
import com.xebialabs.xlrelease.domain.delivery.conditions.ConditionGroup
import com.xebialabs.xlrelease.domain.delivery.{Condition, Delivery, Stage, Transition}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids.{getName, isDeliveryId}
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Page, ReleaseRepository}
import com.xebialabs.xlrelease.service._
import com.xebialabs.xlrelease.utils.CiHelper.rewriteWithNewId
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class DeliveryService @Autowired()(val deliveryRepository: DeliveryRepository,
                                   val releaseRepository: ReleaseRepository,
                                   val releaseService: ReleaseService,
                                   val archivingService: ArchivingService,
                                   val releaseSearchService: ReleaseSearchService,
                                   val ciIdService: CiIdService,
                                   val archivedReleases: ArchivedReleases,
                                   val folderService: FolderService,
                                   val roleService: RoleService,
                                   val eventBus: XLReleaseEventBus)
  extends Logging with DeliveryTimelineCalculator with DeliveryServiceUtils {

  @Timed
  def getDelivery(deliveryId: String): Delivery = {
    val delivery = getDeliveryOrPattern(deliveryId)
    checkIsDelivery(delivery)
    delivery
  }

  @Timed
  def existsDelivery(deliveryId: String): Boolean = deliveryRepository.exists(deliveryId)

  @Timed
  @VisibleForTesting
  def createDelivery(delivery: Delivery): Delivery = createDelivery(delivery, rewriteId = true)

  @Timed
  def createDelivery(delivery: Delivery, rewriteId: Boolean): Delivery = {
    logger.debug(s"Creating new release delivery [$delivery]")
    validateDelivery(delivery)

    val deliveryId = if (rewriteId || delivery.getId == null || !isDeliveryId(delivery.getId)) {
      factory.deliveryId()
    } else {
      delivery.getId
    }

    delivery.setId(deliveryId)

    rewriteWithNewId(delivery, deliveryId)

    delivery.getStages.forEach { stage =>
      Option(stage.getTransition).foreach { transition =>
        transition.setStage(stage)
      }
    }

    val manager = DeliveryStateManager(delivery, ciIdService)
    val startedDelivery = manager.start()

    deliveryRepository.create(delivery)
    eventBus.publish(DeliveryCreatedEvent(delivery))
    manager.getEvents.foreach(eventBus.publish)

    startedDelivery
  }

  @Timed
  def updateDelivery(updated: Delivery): Delivery = {
    logger.debug(s"Updating release delivery [$updated]")
    validateDelivery(updated)
    doUpdate(updated)
  }

  @Timed
  def deleteDelivery(deliveryId: String): Unit = {
    logger.debug(s"Deleting release delivery [$deliveryId]")
    val delivery = deliveryRepository.read(deliveryId)
    checkIsDelivery(delivery)
    doDelete(deliveryId)
  }

  @Timed
  def updateStage(deliveryId: String, updated: Stage): Stage = {
    logger.debug(s"Updating stage '$updated' on delivery '$deliveryId'")
    checkNotNull(updated, "Stage")

    val delivery = getDelivery(deliveryId)
    checkIsUpdatable(delivery)

    val original = delivery.getStageByIdOrTitle(updated.getId)
    val originalBeforeChanges = CiCloneHelper.cloneCi(original)

    original.setOwner(updated.getOwner)
    original.setTeam(updated.getTeam)

    deliveryRepository.update(delivery)
    eventBus.publish(StageUpdatedEvent(originalBeforeChanges, original, delivery))
    original
  }

  @Timed
  def updateTransition(deliveryId: String, updated: Transition): Transition = {
    updateTransition(getDelivery(deliveryId), updated)
  }

  @Timed
  def updateTransition(delivery: Delivery, updated: Transition): Transition = {
    logger.debug(s"Updating transition [$updated]")

    checkIsUpdatable(delivery)
    checkNotNull(updated, "Transition")

    val original = delivery.getTransitionByIdOrTitle(updated.getId)
    val originalBeforeChanges = cloneCi(original)
    val stage = original.getStage
    checkIsUpdatable(stage)
    validateTransition(delivery, stage, updated)

    val updatedConditions = updateConditions(original, updated)
    original.setAutomated(updated.isAutomated)

    deliveryRepository.update(delivery)
    eventBus.publish(TransitionUpdatedEvent(originalBeforeChanges, updated, delivery))
    updatedConditions.foreach(condition => eventBus.publish(ConditionUpdatedEvent(condition, original, delivery)))
    original
  }

  private def updateConditions(originalTransition: Transition, updatedTransition: Transition): Seq[Condition] = {
    val originalConditions = originalTransition.getAllConditions.asScala
    val updatedConditions = updatedTransition.getAllConditions.asScala
    checkArgument(originalConditions.size == updatedConditions.size, "Not permitted to add or remove conditions on a running delivery")
    originalConditions.zip(updatedConditions).flatMap { case (original, updated) =>
      checkArgument(original.getId == updated.getId, s"Updating order or structure of conditions not allowed on a running delivery. " +
        s"Expected condition '${original.getId}', got '${updated.getId}''")
      checkArgument(original.getType == updated.getType, "Condition type cannot be changed on a running delivery")

      original match {
        case _: ConditionGroup => None
        case condition: Condition if condition.isSatisfied => None
        case _ =>
          val changedProps = original.getInputProperties.asScala.filterNot(_.areEqual(original, updated))
          changedProps.foreach(pd => pd.set(original, pd.get(updated)))
          if (changedProps.nonEmpty) Some(original) else None
      }
    }.toSeq
  }

  @Timed
  def search(filters: AbstractDeliveryFilters, page: Page, orderBy: DeliveryOrderMode): java.util.List[Delivery] =
    deliveryRepository.search(filters, page, orderBy, currentPrincipals, currentRoleIds,
      anyOfPermissions = VIEW_DELIVERY_PERMISSION_SET
    ).asJava

  @Timed
  def getTimeline(deliveryId: String, now: DateTime): DeliveryTimeline = {
    val delivery = getDelivery(deliveryId)
    calculateTimeline(delivery, now)
  }

  @Timed
  def getReleases(deliveryId: String): java.util.List[DeliveryFlowReleaseInfo] = {
    val releases = deliveryRepository.findReleasesByDeliveryId(deliveryId)

    val allReleaseIds = deliveryRepository.getByIdOrTitle(deliveryId).getReleaseIds
    val archivedReleases = (allReleaseIds.asScala filterNot (fullId => releases.map(_.getId).contains(getName(fullId)))).toSeq match {
      case Nil => Seq.empty
      case archivedReleaseIds => archivingService.searchReleasesBasicExtByReleaseIds(archivedReleaseIds).map(item =>
        new DeliveryFlowReleaseInfo(getName(item.id), item.title, item.status, item.startDate, item.endDate, true)
      )
    }

    (releases ++ archivedReleases).asJava
  }

  @Timed
  def findDeliveriesReferencingRelease(releaseId: String): Seq[String] = deliveryRepository.findDeliveriesReferencingReleaseId(releaseId)

  @Timed
  def findActiveDeliveriesReferencingFolder(folderId: String): Seq[CiIdWithTitle] = deliveryRepository.findActiveDeliveriesReferencingFolderId(folderId)

  protected override def checkIsUpdatable(existingDelivery: Delivery, action: String = "update"): Unit = {
    checkArgument(existingDelivery.isUpdatable, s"Cannot $action release delivery '${existingDelivery.getTitle}' because it is ${existingDelivery.getStatus}")
  }

  protected def checkIsUpdatable(stage: Stage): Unit = {
    checkArgument(stage.isOpen, s"Cannot update transition on stage '${stage.getTitle}' because it is COMPLETED")
  }
}
