package com.xebialabs.xlrelease.delivery.repository.sql.persistence

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.api.v1.forms.DeliveryOrderMode
import com.xebialabs.xlrelease.api.v1.forms.DeliveryOrderMode.{CREATED_DATE, END_DATE, START_DATE}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.delivery.repository.sql.persistence.DeliverySchema.{RELEASE_DELIVERIES, TRACKED_ITEMS}
import com.xebialabs.xlrelease.domain.delivery.DeliveryStatus
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.repository.sql.persistence.CiId.CiId
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.FOLDERS
import com.xebialabs.xlrelease.repository.sql.persistence.SecurableSqlBuilder
import com.xebialabs.xlrelease.utils.FolderId

object DeliverySqlBuilder {
  val alias: String = "del"
  val folderAlias: String = "f"
  val trackedItemsAlias: String = "ti"

  val STMT_RELEASE_DELIVERY_SELECT: String =
    s"""|SELECT $alias.${RELEASE_DELIVERIES.ID},
        |$alias.${RELEASE_DELIVERIES.CI_UID},
        |$folderAlias.${FOLDERS.FOLDER_PATH},
        |$folderAlias.${FOLDERS.FOLDER_ID},
        |$alias.${RELEASE_DELIVERIES.CONTENT}
        | FROM ${RELEASE_DELIVERIES.TABLE} $alias
        | INNER JOIN ${FOLDERS.TABLE} $folderAlias ON $alias.${RELEASE_DELIVERIES.FOLDER_UID} = $folderAlias.${FOLDERS.CI_UID}""".stripMargin
}

class DeliverySqlBuilder(implicit dialect: Dialect) extends SqlBuilder[DeliverySqlBuilder] with SecurableSqlBuilder {

  import DeliverySqlBuilder._

  def select(): DeliverySqlBuilder = {
    super.select(STMT_RELEASE_DELIVERY_SELECT)
  }

  def withTitleLike(title: String, strict: Boolean = false): DeliverySqlBuilder = {
    if (strict) {
      likeStrict(s"$alias.${RELEASE_DELIVERIES.TITLE}", title)
    } else {
      like(s"$alias.${RELEASE_DELIVERIES.TITLE}", title)
    }
  }

  def withTitle(title: String): DeliverySqlBuilder = {
    if (title != null && !title.isEmpty) {
      conditions += Sql(s"LOWER($alias.${RELEASE_DELIVERIES.TITLE}) = ?", Seq(title.toLowerCase))
    }
    this
  }

  def withTrackedItemTitleLike(title: String, strict: Boolean = false): DeliverySqlBuilder = {
    if (title != null && !title.isEmpty) {
      addJoin(s"INNER JOIN ${TRACKED_ITEMS.TABLE} $trackedItemsAlias ON $alias.${RELEASE_DELIVERIES.CI_UID} = $trackedItemsAlias.${TRACKED_ITEMS.DELIVERY_UID}")

      if (strict) {
        likeStrict(s"$trackedItemsAlias.${TRACKED_ITEMS.TITLE}", title)
      } else {
        like(s"$trackedItemsAlias.${TRACKED_ITEMS.TITLE}", title)
      }
    }
    this
  }

  def withOriginPatternId(patternId: CiId): DeliverySqlBuilder = {
    if (patternId != null && !patternId.isEmpty) {
      conditions += Sql(s"$alias.${RELEASE_DELIVERIES.ORIGIN_PATTERN_ID} = ?", Seq(getName(patternId)))
    }
    this
  }

  def withOneOfStatuses(statuses: Seq[DeliveryStatus]): DeliverySqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"$alias.${RELEASE_DELIVERIES.STATUS} IN (${statuses.map(_ => "?").mkString(",")})", statuses.map(_.value()))
    }
    this
  }

  def withFolder(folderId: CiId): DeliverySqlBuilder = {
    if (folderId != null && !folderId.isEmpty) {
      conditions += Sql(s"$folderAlias.${FOLDERS.FOLDER_ID} = ?", Seq(FolderId(folderId).id))
    }
    this
  }

  def withPermission(permission: Option[Permission], principals: Iterable[String], roleIds: Iterable[String]): DeliverySqlBuilder = {
    if (permission.isDefined) {
      conditions += Sql(
        s"""$folderAlias.${FOLDERS.SECURITY_UID} IN (
           |  ${selectCiIdsWithPermission(principals, roleIds)}
           |)
      """.stripMargin,
        Seq(permission.get.getPermissionName) ++ principals ++ roleIds
      )
    }
    this
  }

  def withPermissions(permissions: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): DeliverySqlBuilder = {
    if (permissions.nonEmpty) {
      val permissionNames = permissions.map(_.getPermissionName)
      conditions += Sql(
        s"""$folderAlias.${FOLDERS.SECURITY_UID} IN (
           |  ${selectCiIdsWithPermissions(permissionNames, principals, roleIds)}
           |)
      """.stripMargin,
        permissionNames ++ principals ++ roleIds
      )
    }
    this
  }

  def orderBy(order: DeliveryOrderMode): DeliverySqlBuilder = {
    if (order != null) {
      order match {
        case START_DATE =>
          orderBy(s"$alias.${RELEASE_DELIVERIES.START_DATE} DESC")
        case END_DATE =>
          orderBy(s"$alias.${RELEASE_DELIVERIES.END_DATE} DESC")
        case CREATED_DATE =>
          orderBy(s"$alias.${RELEASE_DELIVERIES.CI_UID} DESC")
      }
    }
    orderBy(s"$alias.${RELEASE_DELIVERIES.TITLE} ASC")
    this
  }

  override def newInstance: DeliverySqlBuilder = new DeliverySqlBuilder()

}
