package com.xebialabs.xlrelease.delivery.transition.condition.state

import com.xebialabs.xlrelease.delivery.actors.DeliveryActorService
import com.xebialabs.xlrelease.delivery.events.{ItemCompletedEvent, ItemDescopedEvent, ItemRemovedEvent, ItemSkippedEvent}
import com.xebialabs.xlrelease.domain.delivery.conditions.ItemsCompletionCondition
import com.xebialabs.xlrelease.domain.delivery.{Delivery, Stage}
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

@Component
@EventListener
class ItemsConditionExecutionHandler @Autowired()(val deliveryActorService: DeliveryActorService) {

  @Subscribe
  def onItemCompleted(event: ItemCompletedEvent): Unit = tryMarkConditionAsSatisfied(event.delivery, event.stage)

  @Subscribe
  def onItemSkipped(event: ItemSkippedEvent): Unit = tryMarkConditionAsSatisfied(event.delivery, event.stage)

  @Subscribe
  def onItemDescoped(event: ItemDescopedEvent): Unit = tryMarkConditionAsSatisfied(event.delivery)

  @Subscribe
  def onItemRemoved(event: ItemRemovedEvent): Unit = tryMarkConditionAsSatisfied(event.delivery)

  private def tryMarkConditionAsSatisfied(delivery: Delivery): Unit = {
    delivery.getStages.asScala
      .filter(stage => stage.isOpen && stage.hasTransition)
      .foreach(stage => tryMarkConditionAsSatisfied(delivery, stage))
  }

  private def tryMarkConditionAsSatisfied(delivery: Delivery, stage: Stage): Unit = {
    val stageGroup = delivery.getStageGroupOfStage(stage).asScala
    Option(stageGroup.last.getTransition) match {
      case Some(transition) if stageGroup.last.isOpen && transition.isAutomated =>
        transition.getConditionsOfType(classOf[ItemsCompletionCondition]).asScala
          .filter(_.isActive)
          .foreach { condition =>
            if (condition.isSatisfiedBy(stageGroup.toSeq, delivery)) {
              deliveryActorService.markConditionAsSatisfied(delivery.getId, condition.getId)
            }
          }
      case _ => // noop
    }
  }

}
