package com.xebialabs.xlrelease.delivery

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.Metadata.ConfigurationItemRoot
import com.xebialabs.xlrelease.builder.ReleaseBuilder
import com.xebialabs.xlrelease.builder.TaskBuilder.newCustomScript
import com.xebialabs.xlrelease.delivery.util.DeliveryObjectFactory
import com.xebialabs.xlrelease.domain.CustomScriptTask.PYTHON_SCRIPT_PREFIX
import com.xebialabs.xlrelease.domain.delivery.TrackedItemStatus.{NOT_READY, READY, SKIPPED}
import com.xebialabs.xlrelease.domain.delivery._
import com.xebialabs.xlrelease.domain.delivery.conditions.{ItemsCompletionCondition, TimeCondition}
import com.xebialabs.xlrelease.domain.{CustomScriptTask, Release}
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.SEPARATOR
import com.xebialabs.xlrelease.service.CiIdService
import org.joda.time.DateTime

import java.time.{LocalDate, ZoneId}
import java.util.{Date, UUID}
import scala.jdk.CollectionConverters._

object DeliveryTestUtils {

  object TestCiIdService extends CiIdService {
    override def getUniqueId(t: Type, parentId: String): String =
      getUniqueId(t.getName, parentId)

    override def getUniqueId(t: String, parentId: String): String =
      s"$parentId$SEPARATOR$t${UUID.randomUUID().toString.filterNot(_ == '-')}"
  }

  val factory = new DeliveryObjectFactory(TestCiIdService)

  def fromGraph(drawing: String): Delivery = new DeliveryDrawing(drawing).toDelivery

  def delivery(title: String, folderId: String): Delivery = {
    val delivery = emptyDelivery(title)
    delivery.setFolderId(folderId)
    stubStages(delivery)
    delivery
  }

  def emptyDelivery(title: String): Delivery = {
    val delivery = factory.newCiWithId[Delivery](Delivery.DELIVERY_ROOT)
    delivery.setTitle(title)
    delivery.setStartDate(new Date())
    delivery.setEndDate(new DateTime().plusDays(5).toDate)
    delivery.updateDuration()
    delivery
  }

  def stubStages(delivery: Delivery): Unit = {
    Seq(
      new Stage("Feature in Dev"),
      new Stage("Feature in QA"),
      new Stage("Feature in Pre-Prod"),
      new Stage("Feature in Prod")
    ).foreach { stage =>
      stage.setId(factory.stageId(delivery.getId))
      stage.setOwner(stage.getTitle.replace("Feature in", "User"))
      stage.setTeam(stage.getTitle.replace("Feature in", "Team"))
      delivery.addStage(stage)
    }
  }

  def stubTransitions(delivery: Delivery, condition: Condition = null): Unit = {
    delivery.getStages.asScala.take(delivery.getStages.size() - 1).foreach { stage =>
      val transition = factory.newCiWithId[Transition](stage.getId)
      transition.setTitle(s"Wait for ${stage.getTitle}")
      stage.setTransition(transition)
      if (condition != null) {
        val copy = cloneCi(condition)
        copy.getAllConditions.forEach(_.setId(factory.conditionId(transition.getId)))
        transition.addCondition(copy)
      }
    }
  }

  def stubTransitionsWithTimeCondition(delivery: Delivery): Unit = {
    stubTransitions(delivery, {
      val tomorrow = Date.from(LocalDate.now().plusDays(1).atStartOfDay(ZoneId.systemDefault()).toInstant)
      val condition = new TimeCondition
      condition.setFixedDate(tomorrow)
      condition
    })
  }

  def stubTransitionsWithItemsCondition(delivery: Delivery): Unit = {
    stubTransitions(delivery, {
      val condition = new ItemsCompletionCondition
      condition.setIgnoreDescoped(true)
      condition.setStatuses(TrackedItemStatus.DONE_STATUSES.map(_.value()).toList.asJava)
      condition
    })
  }

  def generatePattern(title: String, folderId: String, duration: Int = 0, shouldStubStages: Boolean = false, autoComplete: Boolean = false): Delivery = {
    val pattern = factory.newCiWithId[Delivery](Delivery.DELIVERY_ROOT)
    pattern.setTitle(title)
    pattern.setStatus(DeliveryStatus.TEMPLATE)
    pattern.setPlannedDuration(duration)
    pattern.setFolderId(folderId)
    pattern.setAutoComplete(autoComplete)
    if (shouldStubStages) {
      stubStages(pattern)
    }
    pattern
  }

  def generateStage(patternId: String): Stage = {
    val stage = factory.newCiWithId[Stage](patternId)
    stage.setTitle(Ids.getName(stage.getId))
    stage.setStatus(StageStatus.OPEN)
    stage
  }

  def generateTransition(stageId: String): Transition = {
    val transition = factory.newCiWithId[Transition](stageId)
    transition.setTitle(Ids.getName(transition.getId))
    transition
  }

  def release(title: String, folderId: Option[String] = None): Release = {
    ReleaseBuilder
      .newRelease()
      .withScriptUsername("admin")
      .withScriptUserPassword("admin")
      .withId(
        factory.createUniqueId[Release](folderId.getOrElse(ConfigurationItemRoot.APPLICATIONS.getRootNodeName))
      ).withTitle(title).build
  }

  class DeliveryDrawing(value: String) {
    private def split(value: String, separator: Char): Seq[String] = value.split(separator)
      .toSeq
      .map(_.trim)

    //noinspection ScalaStyle
    def toDelivery: Delivery = {
      val lines = split(value, '\n').filter(_ != "")

      val delivery = emptyDelivery("Graph delivery")

      val (stages, indexes) = split(lines.head, '|')
        .foldLeft((Seq.empty[Stage], Seq.empty[String])) {
          case ((_stages, _indexes), column) =>
            if (column.startsWith("S")) {
              val contentList = column.split('/')
              val stageName = contentList.init.mkString("/")
              val stageStatus = contentList.last

              val stage = factory.newCiWithId[Stage](delivery.getId)
              stage.setTitle(stageName)
              stage.setStatus(StageStatus.valueOf(stageStatus))
              (_stages :+ stage, _indexes :+ stage.getId)
            } else if (column.startsWith("T")) {
              val transition = factory.newCiWithId[Transition](_stages.last.getId)
              transition.setTitle(column)
              _stages.last.setTransition(transition)
              (_stages, _indexes :+ transition.getId)
            } else {
              (_stages, _indexes)
            }
        }

      val trackedItems = lines.tail.map { line =>
        val columns = split(line, '|')

        val trackedItem = factory.newCiWithId[TrackedItem](delivery.getId)
        trackedItem.setCreatedDate(new Date())
        trackedItem.setModifiedDate(new Date())

        columns.head.split('/') match {
          case Array(title, "D") =>
            trackedItem.setTitle(title)
            trackedItem.setDescoped(true)
          case Array(title) =>
            trackedItem.setTitle(title)
          case _ => throw new IllegalArgumentException(s"Unknown syntax for track item name ${columns.head}")
        }

        columns.tail.map {
          case "R" => Some(READY)
          case "NR" => Some(NOT_READY)
          case "S" => Some(SKIPPED)
          case _ => None
        }.zipWithIndex.foreach {
          case (Some(status), index) =>
            val stage = stages.find(_.getId == indexes(index)).get
            val stageItem = factory.newCiWithId[StageTrackedItem](stage.getId)
            stageItem.setTrackedItemId(trackedItem.getId)
            stageItem.setStatus(status)
            stage.addTrackedItems(Seq(stageItem).asJava)
          case _ =>
        }

        trackedItem
      }

      delivery.setStages(stages.asJava)
      delivery.setTrackedItems(trackedItems.asJava)

      delivery
    }
  }

  // TASKS

  object FindOrCreateStrategy extends Enumeration {
    val Latest: FindOrCreateStrategy.Value = Value("Latest delivery from pattern")
    val Pattern: FindOrCreateStrategy.Value = Value("Search by criteria")
  }

  def findOrCreateDeliveryTask(idAndTitle: String,
                               patternId: String,
                               searchStrategy: FindOrCreateStrategy.Value = FindOrCreateStrategy.Latest,
                               titlePattern: String = null,
                               selectionStrategy: String = "Use latest result",
                               fallback: Boolean = false,
                               title: String = null,
                               startDate: Date = null,
                               endDate: Date = null,
                               deliveryIdOutVar: String = null
                              ): CustomScriptTask = {
    val outVar = if (deliveryIdOutVar != null) {
      "${" + deliveryIdOutVar + "}"
    } else {
      null
    }
    generateDeliveryTask(
      "FindOrCreateDelivery",
      idAndTitle,
      inputParameters = Map[String, AnyRef](
        "patternId" -> patternId,
        "searchStrategy" -> searchStrategy.toString,
        "nameFilter" -> titlePattern,
        "selectionStrategy" -> selectionStrategy,
        "fallback" -> fallback.toString,
        "title" -> title,
        "startDate" -> startDate,
        "endDate" -> endDate
      ),
      outputProperties = Map("deliveryId" -> outVar)
    )
  }

  def createDeliveryTask(idAndTitle: String,
                         patternId: String,
                         title: String = null,
                         startDate: Date = null,
                         endDate: Date = null,
                         deliveryIdOutVar: String = null): CustomScriptTask = {
    generateDeliveryTask("CreateDelivery", idAndTitle, Map(
      "patternId" -> patternId,
      "title" -> title,
      "startDate" -> startDate,
      "endDate" -> endDate
    ), Map("deliveryId" -> deliveryIdOutVar))
  }

  def findDeliveryTask(idAndTitle: String,
                       trackedItemsTitlePattern: String = null,
                       deliveryTitlePattern: String = null,
                       selectionStrategy: String = "Use latest result",
                       deliveryIdOutVar: String = null
                      ): CustomScriptTask = {
    val outVar = if (deliveryIdOutVar != null) {
      "${" + deliveryIdOutVar + "}"
    } else {
      null
    }
    generateDeliveryTask(
      "FindDelivery",
      idAndTitle,
      inputParameters = Map[String, AnyRef](
        "itemsFilter" -> trackedItemsTitlePattern,
        "nameFilter" -> deliveryTitlePattern,
        "selectionStrategy" -> selectionStrategy
      ),
      outputProperties = Map("deliveryId" -> outVar)
    )
  }

  def registerTrackedItemsTask(idAndTitle: String, deliveryId: String, trackedItems: Seq[String]): CustomScriptTask = {
    registerTrackedItemsTask(idAndTitle, deliveryId, Right(trackedItems))
  }

  def registerTrackedItemsTask(idAndTitle: String, deliveryId: String, trackedItems: Either[String, Seq[String]]): CustomScriptTask = {
    trackedItems match {
      case Left(listVariable) =>
        generateDeliveryTask("RegisterTrackedItems", idAndTitle, Map[String, AnyRef](
          "deliveryId" -> deliveryId
        ), variableMapping = Map(PYTHON_SCRIPT_PREFIX + "trackedItems" -> ("${" + listVariable + "}")))
      case Right(itemList) =>
        generateDeliveryTask("RegisterTrackedItems", idAndTitle, Map[String, AnyRef](
          "deliveryId" -> deliveryId,
          "trackedItems" -> itemList.asJava
        ))
    }
  }

  def markTrackedItemTask(idAndTitle: String,
                          deliveryId: String,
                          stageId: String,
                          trackedItems: Seq[String],
                          status: TrackedItemStatus = READY,
                          precedingStages: Boolean = false): CustomScriptTask = {
    markTrackedItemTask(idAndTitle, deliveryId, stageId, Right(trackedItems), status, precedingStages)
  }

  def markTrackedItemTask(idAndTitle: String,
                          deliveryId: String,
                          stage: String,
                          trackedItems: Either[String, Seq[String]],
                          status: TrackedItemStatus,
                          precedingStages: Boolean): CustomScriptTask = {
    trackedItems match {
      case Left(listVariable) =>
        generateDeliveryTask("MarkTrackedItems", idAndTitle, Map[String, AnyRef](
          "deliveryId" -> deliveryId,
          "stage" -> stage,
          "status" -> status.toString,
          "precedingStages" -> precedingStages.toString
        ), variableMapping = Map(PYTHON_SCRIPT_PREFIX + "trackedItems" -> ("${" + listVariable + "}")))
      case Right(itemList) =>
        generateDeliveryTask("MarkTrackedItems", idAndTitle, Map[String, AnyRef](
          "deliveryId" -> deliveryId,
          "stage" -> stage,
          "status" -> status.toString,
          "trackedItems" -> itemList.asJava,
          "precedingStages" -> precedingStages.toString
        ))
    }

  }

  def waitForTrackedItemsTask(idAndTitle: String,
                              deliveryId: String,
                              items: Seq[String],
                              stage: String,
                              status: TrackedItemStatus = READY): CustomScriptTask = {
    generateDeliveryTask("WaitForTrackedItems", idAndTitle, Map[String, AnyRef](
      "deliveryId" -> deliveryId,
      "trackedItems" -> items.asJava,
      "stage" -> stage,
      "status" -> status.toString
    ))
  }

  def waitForStageTask(idAndTitle: String, deliveryId: String, stage: String): CustomScriptTask = {
    generateDeliveryTask("WaitForStage", idAndTitle, Map[String, AnyRef](
      "deliveryId" -> deliveryId,
      "stage" -> stage
    ))
  }

  def generateDeliveryTask(`type`: String,
                           idAndTitle: String,
                           inputParameters: Map[String, AnyRef],
                           outputProperties: Map[String, AnyRef] = Map.empty,
                           variableMapping: Map[String, String] = Map.empty): CustomScriptTask = {
    newCustomScript(s"delivery.${`type`}")
      .withIdAndTitle(idAndTitle)
      .withInputParameters(inputParameters.asJava)
      .withOutputProperties(outputProperties.asJava)
      .withVariableMapping(variableMapping.asJava)
      .build()
  }
}
