package com.xebialabs.xlrelease.delivery.actors

import akka.actor.ActorRef
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.xlrelease.api.v1.forms.{CompleteTransition, CreateDelivery, DuplicateDeliveryPattern}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.delivery.actors.DeliveryActor._
import com.xebialabs.xlrelease.domain.delivery._
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import java.util.Optional
import scala.concurrent.{Await, TimeoutException}
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@Component
class DeliveryActorService(deliveryActorHolder: DeliveryActorHolder, xlrConfig: XlrConfig) extends Logging {
  lazy val actorRef: ActorRef = deliveryActorHolder.actorRef()

  def updatePattern(pattern: Delivery): Delivery = askAndAwait {
    UpdatePattern(pattern)
  }

  def deletePattern(deliveryId: String): Unit = askAndAwait {
    DeletePattern(deliveryId)
  }

  def duplicatePattern(deliveryId: String, params: DuplicateDeliveryPattern): Delivery = askAndAwait {
    DuplicatePattern(deliveryId, params)
  }

  def createDeliveryFromPattern(patternId: String, params: CreateDelivery): Delivery = askAndAwait {
    CreateDeliveryFromPattern(patternId, params)
  }

  //

  def updateDelivery(delivery: Delivery): Delivery = askAndAwait {
    UpdateDelivery(delivery)
  }

  def deleteDelivery(deliveryId: String): Unit = askAndAwait {
    DeleteDelivery(deliveryId)
  }

  //

  def createTrackedItem(deliveryId: String, item: TrackedItem): TrackedItem = askAndAwait {
    CreateTrackedItem(deliveryId, item)
  }

  def registerTrackedItems(deliveryId: String, items: java.util.List[String], releaseId: String): Unit = askAndAwait {
    RegisterTrackedItems(deliveryId, items.asScala.toSeq, releaseId)
  }

  def updateTrackedItem(deliveryId: String, item: TrackedItem): TrackedItem = askAndAwait {
    UpdateTrackedItem(deliveryId, item)
  }

  def deleteTrackedItem(deliveryId: String, itemId: String): Unit = askAndAwait {
    DeleteTrackedItem(deliveryId, itemId)
  }

  def descopeTrackedItem(deliveryId: String, itemId: String): Unit = askAndAwait {
    DescopeTrackedItem(deliveryId, itemId)
  }

  def rescopeTrackedItem(deliveryId: String, itemId: String): Unit = askAndAwait {
    RescopeTrackedItem(deliveryId, itemId)
  }

  def skipTrackedItem(deliveryId: String, stageId: String, itemId: String): Unit = askAndAwait {
    SkipTrackedItem(deliveryId, stageId, itemId)
  }

  def resetTrackedItem(deliveryId: String, stageId: String, itemId: String): Unit = askAndAwait {
    ResetTrackedItem(deliveryId, stageId, itemId)
  }

  def markTrackedItemsInStage(deliveryId: String,
                              stageId: String,
                              items: java.util.List[String],
                              status: TrackedItemStatus,
                              precedingStages: Boolean,
                              fromReleaseId: String): java.util.List[TrackedItem] = askAndAwait {
    MarkTrackedItemsInStage(deliveryId, stageId, items.asScala.toSeq, status, precedingStages, fromReleaseId)
  }

  //

  def addStage(patternId: String, stage: Stage, position: Optional[Integer]): Stage = askAndAwait {
    AddStage(patternId, stage, position)
  }

  def addStageBetween(patternId: String, stage: Stage, before: Optional[String], after: Optional[String]): Stage = askAndAwait {
    AddStageBetween(patternId, stage, before.asScala, after.asScala)
  }

  def updateStage(patternId: String, stage: Stage, fromBatch: Boolean = false): Stage = askAndAwait {
    UpdateStage(patternId, stage, fromBatch)
  }

  def deleteStage(patternId: String, stageId: String): Unit = askAndAwait {
    DeleteStage(patternId, stageId)
  }

  def completeStage(deliveryId: String, stageIdOrTitle: String): Unit = askAndAwait {
    CompleteStage(deliveryId, stageIdOrTitle)
  }

  def reopenStage(deliveryId: String, stageIdOrTitle: String): Unit = askAndAwait {
    ReopenStage(deliveryId, stageIdOrTitle)
  }

  //

  def addTransition(patternId: String, stageIdOrTitle: String, transition: Transition): Transition = askAndAwait {
    AddTransition(patternId, stageIdOrTitle, transition)
  }

  def updateTransition(patternId: String, transition: Transition): Transition = askAndAwait {
    UpdateTransition(patternId, transition)
  }

  def deleteTransition(patternId: String, transitionId: String): Unit = askAndAwait {
    DeleteTransition(patternId, transitionId)
  }

  //

  def markConditionAsSatisfied(deliveryId: String, conditionId: String): Unit = tell {
    val command = MarkConditionAsSatisfied(deliveryId, conditionId)
    command.callerContext = None
    command
  }

  //

  def registerSubscriber(deliveryId: String, subscriber: Subscriber): SubscriptionResult = askAndAwait {
    RegisterSubscriber(deliveryId, subscriber)
  }

  //

  def completeTransition(deliveryId: String, transitionId: String, parameters: CompleteTransition): Unit = askAndAwait {
    ManualCompleteTransition(deliveryId, transitionId, parameters)
  }

  private def askAndAwait[T](msg: AnyRef): T = {
    implicit val askTimeout: Timeout = xlrConfig.timeouts.releaseActionResponse
    try {
      Await.result(
        actorRef ? msg,
        askTimeout.duration
      ).asInstanceOf[T]
    } catch {
      case e: TimeoutException => throw new RuntimeException(e)
      case e: InterruptedException => throw new RuntimeException(e)
    }
  }

  private def tell(msg: AnyRef): Unit = actorRef ! msg

}
