package com.xebialabs.xlrelease.delivery.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.{isNotBlank, isNotEmpty}
import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.xlrelease.db.{ArchivedReleases, DbConstants}
import com.xebialabs.xlrelease.delivery.events.{DeliveryDeletedEvent, DeliveryUpdatedEvent}
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.util.DeliveryObjectFactory
import com.xebialabs.xlrelease.domain.delivery.{Delivery, Stage, Transition}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.isRoot
import com.xebialabs.xlrelease.service.{CiIdService, FolderService, ReleaseService}
import grizzled.slf4j.Logging
import org.apache.commons.lang3.time.DateUtils

import java.time.temporal.ChronoUnit
import java.time.{Instant, LocalDateTime, ZoneId}
import java.util.Date
import scala.collection.mutable
import scala.jdk.CollectionConverters._

trait DeliveryServiceUtils extends Logging {
  val deliveryRepository: DeliveryRepository
  val eventBus: XLReleaseEventBus
  val releaseService: ReleaseService
  val archivedReleases: ArchivedReleases
  val folderService: FolderService
  val roleService: RoleService
  val ciIdService: CiIdService

  protected val factory = new DeliveryObjectFactory(ciIdService)

  protected def checkIsUpdatable(existingDelivery: Delivery, action: String = "update"): Unit

  @Timed
  def getDeliveryOrPattern(deliveryId: String): Delivery = deliveryRepository.read(deliveryId)

  @Timed
  def getFolderId(deliveryId: String): String = deliveryRepository.findFolderId(deliveryId)

  protected def doUpdate(updated: Delivery): Delivery = {
    val original = deliveryRepository.read(updated.getId)
    checkIsUpdatable(original)
    original.setTitle(updated.getTitle)
    original.setDescription(updated.getDescription)
    original.setAutoComplete(updated.isAutoComplete)
    computeAndUpdateDates(original, updated.getPlannedDuration, updated.getStartDate, updated.getEndDate)
    if (isNotBlank(updated.getFolderId)) {
      original.setFolderId(updated.getFolderId)
    }

    deliveryRepository.update(original)
    eventBus.publish(DeliveryUpdatedEvent(original))
    original
  }

  protected def doDelete(deliveryId: String): Unit = {
    if (deliveryRepository.exists(deliveryId)) {
      val delivery = deliveryRepository.read(deliveryId)
      checkIsUpdatable(delivery, "delete")
      deliveryRepository.delete(deliveryId)
      eventBus.publish(DeliveryDeletedEvent(delivery))
    } else {
      logger.debug(s"Release delivery [$deliveryId] already deleted")
    }
  }

  protected def validate(delivery: Delivery): Unit = {
    validate(delivery, true)
  }

  protected def validate(delivery: Delivery, withQueries: Boolean): Unit = {
    checkNotNull(delivery, "Release delivery")
    checkArgument(isNotBlank(delivery.getTitle), "Title must be set")
    checkArgument(delivery.getTitle.length < 256, "Title must be 255 characters or less")
    checkArgument(!delivery.getTitle.contains(DbConstants.ESCAPE_CHAR), s"Title must not contain reserved character ${DbConstants.ESCAPE_CHAR}")
    checkArgument(isNotBlank(delivery.getFolderId), "Folder ID must be set")
    checkArgument(!isRoot(delivery.getFolderId), s"Provided folder ID '${delivery.getFolderId}' must not be a root folder")
    if (withQueries) checkArgument(folderService.exists(delivery.getFolderId), s"Provided folder ID '${delivery.getFolderId}' must exist in the database")
    if (isNotEmpty(delivery.getDescription)) checkArgument(delivery.getDescription.length < 1024, "Description must be 1024 characters or less")
  }

  protected def validateDelivery(delivery: Delivery): Unit = {
    validate(delivery)
    checkIsDelivery(delivery)
    checkArgument(delivery.getStartDate != null, "Start date must be set")
    checkArgument(delivery.getEndDate != null, "End date must be set")
    checkArgument(delivery.getEndDate.after(delivery.getStartDate), "End date must be after start date")
  }

  protected def validateTransition(delivery: Delivery, stage: Stage, transition: Transition): Unit = {
    checkNotNull(transition, "Transition")
    checkArgument(isNotBlank(transition.getTitle), "Transition title must be set")
    checkArgument(transition.getTitle.length < 256, "Transition title must be 255 characters or less")
    checkArgument(!transition.getTitle.contains(DbConstants.ESCAPE_CHAR), s"Title must not contain reserved character ${DbConstants.ESCAPE_CHAR}")
    checkArgument(!delivery.isLastStage(stage), "Transition can not be added to the last stage")
    transition.getAllConditions.asScala.foreach(_.validate(delivery))
  }

  protected def checkIsDelivery(delivery: Delivery): Unit = {
    checkArgument(!delivery.isTemplate, s"${delivery.getId} must be a Delivery")
  }

  protected def currentPrincipals: Iterable[String] = authenticationToPrincipals(getAuthentication).asScala

  protected def currentRoleIds: mutable.Buffer[String] = roleService.getRolesFor(getAuthentication).asScala.map(_.getId)

  protected def computeAndUpdateDates(delivery: Delivery, durationOpt: Int = 0, startDateOpt: Date = null, endDateOpt: Date = null): Unit = {
    val isNotEmptyDate = (date: Date) => Option(date).exists(_.getTime != 0)
    val plannedDuration = Option(durationOpt).filter(_ > 0).getOrElse(delivery.getPlannedDuration.toInt)

    val (duration, startDate, endDate) = if (isNotEmptyDate(startDateOpt) && isNotEmptyDate(endDateOpt)) {
      (calculateDuration(startDateOpt, endDateOpt), startDateOpt, endDateOpt)
    } else if (isNotEmptyDate(startDateOpt)) {
      (plannedDuration, startDateOpt, calculateOtherDate(startDateOpt, plannedDuration))
    } else if (isNotEmptyDate(endDateOpt)) {
      (plannedDuration, calculateOtherDate(endDateOpt, plannedDuration, -1), endDateOpt)
    } else {
      val startDate = Date.from(Instant.now())
      (plannedDuration, startDate, calculateOtherDate(startDate, plannedDuration))
    }

    delivery.setPlannedDuration(duration)
    delivery.setStartDate(startDate)
    delivery.setEndDate(endDate)
  }

  private def calculateOtherDate(date: Date, duration: Int, addition: Int = 1): Date = if (duration > 0) {
    DateUtils.addHours(date, duration * addition)
  } else {
    DateUtils.addMonths(date, 1 * addition)
  }

  private def calculateDuration(startDate: Date, endDate: Date): Int = {
    val localScheduledStartDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(startDate.getTime), ZoneId.systemDefault)
    val localDueDate = LocalDateTime.ofInstant(Instant.ofEpochMilli(endDate.getTime), ZoneId.systemDefault)
    localScheduledStartDate.until(localDueDate, ChronoUnit.HOURS).toInt
  }
}
