package com.xebialabs.xlrelease.delivery.actors

import akka.actor.{Actor, ActorLogging, Props, ReceiveTimeout}
import akka.cluster.sharding.ShardRegion
import com.xebialabs.xlrelease.api.v1.forms.{CompleteTransition, CreateDelivery, DuplicateDeliveryPattern}
import com.xebialabs.xlrelease.delivery.actors.DeliveryActor._
import com.xebialabs.xlrelease.delivery.service.{DeliveryExecutionService, DeliveryPatternService, DeliveryService}
import com.xebialabs.xlrelease.domain.delivery._
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder

import java.util.Optional
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

object DeliveryActor {
  def props(clustered: Boolean,
            deliveryService: DeliveryService,
            deliveryPatternService: DeliveryPatternService,
            deliveryExecutionService: DeliveryExecutionService) =
    Props(new DeliveryActor(clustered, deliveryService, deliveryPatternService, deliveryExecutionService)).withDispatcher("xl.dispatchers.release-dispatcher")


  trait DeliveryAction {
    def deliveryId: String
  }

  abstract class DeliveryCommand extends DeliveryAction with Serializable {
    var callerContext: Option[Authentication] = Option(SecurityContextHolder.getContext.getAuthentication)
  }

  case class UpdatePattern(pattern: Delivery) extends DeliveryCommand {
    override def deliveryId: String = pattern.getId
  }

  case class DeletePattern(deliveryId: String) extends DeliveryCommand

  case class DuplicatePattern(deliveryId: String, params: DuplicateDeliveryPattern) extends DeliveryCommand

  case class CreateDeliveryFromPattern(deliveryId: String, params: CreateDelivery) extends DeliveryCommand

  case class UpdateDelivery(delivery: Delivery) extends DeliveryCommand {
    override def deliveryId: String = delivery.getId
  }

  case class DeleteDelivery(deliveryId: String) extends DeliveryCommand

  case class RegisterSubscriber(deliveryId: String, subscriber: Subscriber) extends DeliveryCommand

  case class CreateTrackedItem(deliveryId: String, item: TrackedItem) extends DeliveryCommand

  case class RegisterTrackedItems(deliveryId: String, items: Seq[String], releaseId: String) extends DeliveryCommand

  case class UpdateTrackedItem(deliveryId: String, item: TrackedItem) extends DeliveryCommand

  case class DeleteTrackedItem(deliveryId: String, itemId: String) extends DeliveryCommand

  case class DescopeTrackedItem(deliveryId: String, itemId: String) extends DeliveryCommand

  case class RescopeTrackedItem(deliveryId: String, itemId: String) extends DeliveryCommand

  case class SkipTrackedItem(deliveryId: String, stageId: String, itemId: String) extends DeliveryCommand

  case class ResetTrackedItem(deliveryId: String, stageId: String, itemId: String) extends DeliveryCommand

  case class MarkTrackedItemsInStage(deliveryId: String,
                                     stageId: String,
                                     items: Seq[String],
                                     status: TrackedItemStatus,
                                     precedingStages: Boolean,
                                     releaseId: String) extends DeliveryCommand

  case class AddStage(deliveryId: String, stage: Stage, position: Optional[Integer]) extends DeliveryCommand

  case class AddStageBetween(deliveryId: String, stage: Stage, before: Option[String], after: Option[String]) extends DeliveryCommand

  case class UpdateStage(deliveryId: String, stage: Stage, fromBatch: Boolean) extends DeliveryCommand

  case class DeleteStage(deliveryId: String, stageId: String) extends DeliveryCommand

  case class CompleteStage(deliveryId: String, stageId: String) extends DeliveryCommand

  case class ReopenStage(deliveryId: String, stageId: String) extends DeliveryCommand

  case class ManualCompleteTransition(deliveryId: String, transitionId: String, parameters: CompleteTransition) extends DeliveryCommand

  case class AddTransition(deliveryId: String, stageIdOrTitle: String, transition: Transition) extends DeliveryCommand

  case class UpdateTransition(deliveryId: String, transition: Transition) extends DeliveryCommand

  case class DeleteTransition(deliveryId: String, transitionId: String) extends DeliveryCommand

  case class MarkConditionAsSatisfied(deliveryId: String, conditionId: String) extends DeliveryCommand

}

class DeliveryActor(clustered: Boolean,
                    deliveryService: DeliveryService,
                    deliveryPatternService: DeliveryPatternService,
                    deliveryExecutionService: DeliveryExecutionService)
  extends Actor with ActorLogging {

  context.setReceiveTimeout(30.minutes)

  override def receive: Actor.Receive = withCallerContext(manageDeliveries orElse trackItems orElse manageStages orElse
    manageTransitions orElse manageConditions orElse managePatterns orElse passivate)

  //noinspection ScalaStyle
  private def trackItems: Actor.Receive = {
    case CreateTrackedItem(deliveryId, item) => replyOrFail {
      deliveryExecutionService.createTrackedItem(deliveryId, item)
    }
    case RegisterTrackedItems(deliveryId, items, releaseId) => replyOrFail {
      deliveryExecutionService.registerTrackedItems(deliveryId, items, releaseId)
    }
    case UpdateTrackedItem(deliveryId, item) => replyOrFail {
      deliveryExecutionService.updateTrackedItem(deliveryId, item)
    }
    case DeleteTrackedItem(deliveryId, itemId) => replyOrFail {
      deliveryExecutionService.deleteTrackedItem(deliveryId, itemId)
    }
    case DescopeTrackedItem(deliveryId, itemId) => replyOrFail {
      deliveryExecutionService.descopeTrackedItem(deliveryId, itemId)
    }
    case RescopeTrackedItem(deliveryId, itemId) => replyOrFail {
      deliveryExecutionService.rescopeTrackedItem(deliveryId, itemId)
    }
    case SkipTrackedItem(deliveryId, stageId, itemId) => replyOrFail {
      deliveryExecutionService.skipTrackedItemInStage(deliveryId, stageId, itemId)
    }
    case MarkTrackedItemsInStage(deliveryId, stageId, items, status, precedingStages, releaseId) => replyOrFail {
      deliveryExecutionService.markTrackedItemsInStage(deliveryId, stageId, items, status, releaseId, precedingStages)
    }
    case ResetTrackedItem(deliveryId, stageId, itemId) => replyOrFail {
      deliveryExecutionService.resetTrackedItemInStage(deliveryId, stageId, itemId)
    }
    case RegisterSubscriber(deliveryId, subscriber) => replyOrFail {
      deliveryExecutionService.registerSubscriber(deliveryId, subscriber)
    }
  }

  private def manageStages: Actor.Receive = {
    case AddStage(deliveryId, stage, position) => replyOrFail {
      deliveryPatternService.addStage(deliveryId, stage, position)
    }
    case AddStageBetween(deliveryId, stage, before, after) => replyOrFail {
      deliveryPatternService.addStageBetween(deliveryId, stage, before, after)
    }
    case UpdateStage(deliveryId, stage, fromBatch) => replyOrFail {
      val delivery = deliveryService.getDeliveryOrPattern(deliveryId)
      if (delivery.isTemplate) {
        deliveryPatternService.updateStage(deliveryId, stage, fromBatch)
      } else {
        deliveryService.updateStage(deliveryId, stage)
      }
    }
    case DeleteStage(deliveryId, stageId) => replyOrFail {
      deliveryPatternService.deleteStage(deliveryId, stageId)
    }
    case CompleteStage(deliveryId, stageId) => replyOrFail {
      deliveryExecutionService.completeStage(deliveryId, stageId)
    }
    case ReopenStage(deliveryId, stageId) => replyOrFail {
      deliveryExecutionService.reopenStage(deliveryId, stageId)
    }
  }

  private def manageTransitions: Actor.Receive = {
    case ManualCompleteTransition(deliveryId, transitionId, parameters) => replyOrFail {
      deliveryExecutionService.manualCompleteTransition(deliveryId, transitionId, parameters)
    }
    case AddTransition(deliveryId, stageIdOrTitle, transition) => replyOrFail {
      deliveryPatternService.addTransition(deliveryId, stageIdOrTitle, transition)
    }
    case UpdateTransition(deliveryId, transition) => replyOrFail {
      val delivery = deliveryService.getDeliveryOrPattern(deliveryId)
      if (delivery.isTemplate) {
        deliveryPatternService.updateTransition(delivery, transition)
      } else {
        deliveryService.updateTransition(delivery, transition)
      }
    }
    case DeleteTransition(deliveryId, transitionId) => replyOrFail {
      deliveryPatternService.deleteTransition(deliveryId, transitionId)
    }
  }

  private def manageConditions: Actor.Receive = {
    case MarkConditionAsSatisfied(deliveryId, conditionId) =>
      deliveryExecutionService.markConditionAsSatisfied(deliveryId, conditionId)
  }

  private def manageDeliveries: Actor.Receive = {
    case CreateDeliveryFromPattern(patternId, params) => replyOrFail {
      deliveryPatternService.createDeliveryFromPattern(patternId, params)
    }
    case UpdateDelivery(delivery) => replyOrFail {
      deliveryService.updateDelivery(delivery)
    }
    case DeleteDelivery(deliveryId) => replyOrFail {
      deliveryService.deleteDelivery(deliveryId)
    }
  }

  private def managePatterns: Actor.Receive = {
    case UpdatePattern(pattern) => replyOrFail {
      deliveryPatternService.updateDeliveryPattern(pattern)
    }
    case DeletePattern(deliveryId) => replyOrFail {
      deliveryPatternService.deleteDeliveryPattern(deliveryId)
    }
    case DuplicatePattern(deliveryId, params) => replyOrFail {
      deliveryPatternService.duplicateDeliveryPattern(deliveryId, params)
    }
  }

  private def passivate: Actor.Receive = {
    case ReceiveTimeout if clustered => context.parent ! ShardRegion.Passivate
    case ReceiveTimeout if !clustered => context.stop(self)
    case ShardRegion.Passivate => context.stop(self)
  }

  private def replyOrFail[T](call: => T): Unit = sender() ! (Try(call) match {
    case Success(t) if t != null => t
    case Success(_) => akka.actor.Status.Failure(new NullPointerException("Method returned null and this cannot be processed"))
    case Failure(ex) =>
      log.error(ex, "Failed to process release delivery message")
      akka.actor.Status.Failure(ex)
  })

  private def withCallerContext(delegate: Receive): Receive = {
    case cmd: DeliveryCommand =>
      val maybeAuthentication = cmd.callerContext
      log.debug(s"Setting caller as ${maybeAuthentication.map(_.getName)} for command $cmd")
      SecurityContextHolder.getContext.setAuthentication(maybeAuthentication.orNull)
      try {
        delegate(cmd)
      } finally {
        SecurityContextHolder.clearContext()
      }
    case msg@_ =>
      delegate(msg)
  }

}
