package com.xebialabs.xlrelease.delivery.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.v1.forms.CompleteTransition
import com.xebialabs.xlrelease.db.DbConstants
import com.xebialabs.xlrelease.delivery.events.ConditionSatisfiedEvent
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.transition.{ConditionTrigger, TransitionParams, UserTrigger}
import com.xebialabs.xlrelease.domain.delivery._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.utils.DeliveryUtils._
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.Ids.{getName, isDomainId, isReleaseId, isTaskId}
import com.xebialabs.xlrelease.repository.{ReleaseRepository, TaskRepository}
import com.xebialabs.xlrelease.service.CiIdService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{Date, List => JList}
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@Service
class DeliveryExecutionService @Autowired()(deliveryRepository: DeliveryRepository,
                                            releaseRepository: ReleaseRepository,
                                            taskRepository: TaskRepository,
                                            ciIdService: CiIdService,
                                            eventBus: XLReleaseEventBus,
                                            releaseActorService: ReleaseActorService) extends Logging {

  // TRANSITIONS
  @Timed
  def markConditionAsSatisfied(deliveryId: String, conditionId: String): Unit = {
    logger.debug(s"Marking condition '$conditionId' as satisfied")
    val delivery = deliveryRepository.read(deliveryId)
    val transition = delivery.getTransitionByIdOrTitle(transitionIdFrom(conditionId))
    val condition = transition.getConditionById(conditionId)
    val stage = transition.getStage

    val root = transition.getRootCondition
    val changes = root.markAsSatisfied(conditionId, new Date()).asScala -= root

    if (changes.nonEmpty) {
      deliveryRepository.update(delivery)
      changes.map(ConditionSatisfiedEvent(_, transition, delivery)).foreach(eventBus.publish)

      if (root.isSatisfied && stage.isOpen) {
        val stages = delivery.getStageGroupOfStage(stage).asScala.toSeq
        val completedItems = getItemsCompletedInAllStages(delivery.getTrackedItems.asScala.filterNot(_.isDescoped).toSeq, stages).map(_.getId).toSet
        val transitionedItems = delivery.findNextStage(stage).asScala.map(_.getItems.asScala.map(_.getTrackedItemId)).getOrElse(Seq.empty)
        val toTransition = completedItems -- transitionedItems
        executeTransition(transition, TransitionParams(delivery, stage, toTransition, closeStages = true, ConditionTrigger(conditionId, condition.toString)))
      }
    }
  }

  @Timed
  def manualCompleteTransition(deliveryId: String, transitionId: String, parameters: CompleteTransition): Unit = {
    logger.debug(s"Manual transition '$transitionId' on delivery '$deliveryId' with params '$parameters'")
    val delivery = deliveryRepository.read(deliveryId)
    val stage = delivery.getStageByTransitionId(transitionId)
    val transition = stage.getTransition

    checkArgument(stage.isOpen, s"Transition already completed on stage '${stage.getTitle}'")
    checkNotNull(parameters.getTransitionItems, "Transition items")
    val transitionItems = parameters.getTransitionItems.asScala.toSet
    transitionItems.foreach { itemId =>
      val trackedItem = delivery.getItemByIdOrTitle(itemId) // tracked item exists in delivery
      checkArgument(!trackedItem.isDescoped, s"De-scoped tracked item '${trackedItem.getTitle}' can not be transitioned")

      val notDoneInStages = (delivery.getStagesBefore(stage).asScala :+ stage).filterNot(_.getItemById(itemId).getStatus.isDone)
      checkArgument(notDoneInStages.isEmpty, s"Tracked item '${trackedItem.getTitle}' must be completed or skipped" +
        s" in stage '${notDoneInStages.map(_.getTitle).mkString("', '")}' in order to transition to the next stage")
    }

    executeTransition(transition, TransitionParams(delivery, stage, transitionItems, parameters.isCloseStages, UserTrigger(getAuthenticatedUserName)))
  }

  private def executeTransition(transition: Transition, params: TransitionParams): Unit = {
    val manager = DeliveryStateManager(params.delivery, ciIdService)
    manager.handleTransition(transition, params)
    processChanges(manager)
  }

  // SUBSCRIBERS

  @Timed
  def registerSubscriber(deliveryId: String, subscriber: Subscriber): SubscriptionResult = {
    logger.debug(s"Registering subscriber '$subscriber' to delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)
    validateTask(subscriber.sourceId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    val result = manager.registerSubscriber(subscriber)
    processChanges(manager)
    result
  }

  // TRACKED ITEMS

  @Timed
  def getTrackedItemByTitle(deliveryId: String, itemTitle: String): TrackedItem =
    deliveryRepository.read(deliveryId).getItemByIdOrTitle(itemTitle)

  @Timed
  def getTrackedItems(deliveryId: String): JList[TrackedItem] = {
    val delivery = deliveryRepository.read(deliveryId)

    delivery.getTrackedItems
  }

  @Timed
  def createTrackedItem(deliveryId: String, item: TrackedItem): TrackedItem = {
    logger.debug(s"Creating new tracked item '${item.getTitle}' on delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)
    validateItem(delivery, item)

    val manager = DeliveryStateManager(delivery, ciIdService)
    val addedItem = manager.addTrackedItem(item)
    processChanges(manager)
    addedItem
  }

  @Timed
  def updateTrackedItem(deliveryId: String, item: TrackedItem): TrackedItem = {
    logger.debug(s"Updating tracked item with id '${item.getId}' on delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)
    validateItem(delivery, item)

    val manager = DeliveryStateManager(delivery, ciIdService)
    val updatedItem = manager.updateTrackedItem(item)
    processChanges(manager)
    updatedItem
  }

  @Timed
  def deleteTrackedItem(deliveryId: String, itemId: String): Unit = {
    logger.debug(s"Removing tracked item '$itemId' from delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    manager.deleteTrackedItem(itemId)
    processChanges(manager)
  }

  @Timed
  def descopeTrackedItem(deliveryId: String, itemId: String): Unit = {
    logger.debug(s"Descoping tracked item '$itemId' from delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    manager.descopeItems(Seq(delivery.getItemByIdOrTitle(itemId)))
    processChanges(manager)
  }

  @Timed
  def rescopeTrackedItem(deliveryId: String, itemId: String): Unit = {
    logger.debug(s"Rescoping tracked item '$itemId' from delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    manager.rescopeItems(Seq(delivery.getItemByIdOrTitle(itemId)))
    processChanges(manager)
  }

  @Timed
  def registerTrackedItems(deliveryId: String, items: Seq[String], fromReleaseId: String): Unit = {
    logger.debug(s"Registering new tracked items [${items.mkString(", ")}] on delivery '$deliveryId' from release '$fromReleaseId'")
    validateMember(fromReleaseId)
    val delivery = deliveryRepository.read(deliveryId)

    val (existingItems, newItems) = partitionItemsByExistingAndNew(delivery, items.toSet)

    val manager = DeliveryStateManager(delivery, ciIdService)
    delivery.addReleaseId(fromReleaseId)
    existingItems.foreach { existingItem =>
      existingItem.addReleaseId(fromReleaseId)
      addReleaseIdToFirstOpenStage(delivery, existingItem.getId, fromReleaseId)
    }

    newItems.foreach { item =>
      item.addReleaseId(fromReleaseId)
      logger.debug(s"Creating new tracked item '${item.getTitle}' on delivery '$deliveryId'")
      // member already validated
      validateItem(delivery, item, validateMembers = false)
      manager.addTrackedItem(item)
      addReleaseIdToFirstOpenStage(delivery, item.getId, fromReleaseId)
    }
    processChanges(manager)
  }

  @Timed
  def markTrackedItemsInStage(deliveryId: String,
                              stageId: String,
                              items: Seq[String],
                              status: TrackedItemStatus,
                              fromReleaseId: String,
                              precedingStages: Boolean = false): JList[TrackedItem] = {
    logger.debug(
      s"Marking tracked items [${items.mkString(", ")}] as completed on stage '$stageId' on delivery '$deliveryId'"
        + s"${if (fromReleaseId != null) s"from '$fromReleaseId'" else ""}"
    )

    val delivery = deliveryRepository.read(deliveryId)
    val stage = delivery.getStageByIdOrTitle(stageId)

    val existingItems = items.distinct.map { idOrTitle =>
      lazy val lowercaseTitle = idOrTitle.toLowerCase()
      delivery.getTrackedItems.asScala
        .find(trackedItem => getName(trackedItem.getId) == getName(idOrTitle) || trackedItem.getTitle.toLowerCase() == lowercaseTitle)
        .getOrElse(throw new LogFriendlyNotFoundException(s"Tracked item '$idOrTitle' does not exist in delivery '${delivery.getTitle}'"))
    }

    val manager = DeliveryStateManager(delivery, ciIdService)

    if (fromReleaseId != null) {
      validateMember(fromReleaseId)
      existingItems.foreach(manager.addReleaseToItem(_, fromReleaseId))
    }

    val markedItems = manager.markTrackedItemsInStage(existingItems, stage, status, precedingStages, fromReleaseId)
    processChanges(manager)
    markedItems.asJava
  }

  @Timed
  def skipTrackedItemInStage(deliveryId: String, stageId: String, itemId: String): Unit = {
    logger.debug(s"Skipping tracked item '$itemId' on stage '$stageId' in delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)
    val stage = delivery.getStageByIdOrTitle(stageId)

    val trackedItem = delivery.getItemByIdOrTitle(itemId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    manager.skipTrackedItemInStage(trackedItem, stage)
    processChanges(manager)
  }

  @Timed
  def resetTrackedItemInStage(deliveryId: String, stageId: String, itemId: String): Unit = {
    logger.debug(s"Resetting tracked item '$itemId' on stage '$stageId' in delivery '$deliveryId'")
    val delivery = deliveryRepository.read(deliveryId)
    val stage = delivery.getStageByIdOrTitle(stageId)

    val trackedItem = delivery.getItemByIdOrTitle(itemId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    manager.resetTrackedItemInStage(trackedItem, stage)
    processChanges(manager)
  }
  // STAGES

  @Timed
  def completeStage(deliveryId: String, stageId: String): Unit = {
    logger.debug(s"Marking stage '$stageId' as completed")
    doWithStage(deliveryId, stageId, (manager, stage) => manager.completeStage(stage))
  }

  @Timed
  def reopenStage(deliveryId: String, stageId: String): Unit = {
    logger.debug(s"Marking stage '$stageId' as open again")
    doWithStage(deliveryId, stageId, (manager, stage) => manager.reopenStage(stage))
  }

  private def doWithStage(deliveryId: String, stageId: String, fn: (DeliveryStateManager, Stage) => Unit): Unit = {
    val delivery = deliveryRepository.read(deliveryId)
    val stage = delivery.getStageByIdOrTitle(stageId)

    val manager = DeliveryStateManager(delivery, ciIdService)
    fn(manager, stage)
    processChanges(manager)
  }

  private def processChanges(manager: DeliveryStateManager): Unit = {
    manager.completeDeliveryIfNecessary()

    deliveryRepository.update(manager.delivery)
    manager.getMessages.foreach(logger.info(_))
    manager.getEvents.foreach(eventBus.publish)
  }

  private def partitionItemsByExistingAndNew(delivery: Delivery, itemIdOrTitles: Set[String]): (Seq[TrackedItem], Seq[TrackedItem]) = {
    val idsByTitles = Map(delivery.getTrackedItems.asScala.map(i => (i.getTitle.toLowerCase(), getName(i.getId))).toSeq: _*)
    val itemsByIds = Map(delivery.getTrackedItems.asScala.map(i => (getName(i.getId), i)).toSeq: _*)

    itemIdOrTitles.foldRight(Seq.empty[TrackedItem] -> Seq.empty[TrackedItem]) { case (idOrTitle, (existingAcc, newAcc)) =>
      itemsByIds.get(getName(idOrTitle))
        .orElse(idsByTitles.get(idOrTitle.toLowerCase()).flatMap(itemsByIds.get))
        .fold(existingAcc -> (new TrackedItem(idOrTitle) +: newAcc))(item => (item +: existingAcc) -> newAcc)
    }
  }

  private def addReleaseIdToFirstOpenStage(delivery: Delivery, itemId: String, releaseId: String): Unit = {
    delivery.findFirstOpenStage().asScala.foreach { stage =>
      stage.findItemById(itemId).asScala.foreach(_.addReleaseId(releaseId))
    }
  }

  private def validateItem(delivery: Delivery, trackedItem: TrackedItem, validateMembers: Boolean = true): Unit = {
    checkNotNull(trackedItem, "Tracked item")
    checkArgument(isNotBlank(trackedItem.getTitle), "Tracked item title must be set")
    checkArgument(trackedItem.getTitle.length < 256, "Title must be 255 characters or less")
    checkArgument(!trackedItem.getTitle.contains(DbConstants.ESCAPE_CHAR), s"Title must not contain reserved character ${DbConstants.ESCAPE_CHAR}")
    lazy val lowerCaseTitle = trackedItem.getTitle.toLowerCase()
    checkArgument(!delivery.getTrackedItems.asScala.exists(existing => existing.getTitle.toLowerCase() == lowerCaseTitle && existing.getId != trackedItem.getId),
      s"A tracked item with title '${trackedItem.getTitle}' already exists")
    if (trackedItem.getReleaseIds == null) {
      trackedItem.setReleaseIds(new java.util.HashSet[String])
    }
    if (validateMembers) {
      trackedItem.getReleaseIds.forEach(validateMember)
    }
  }

  private def validateMember(releaseId: String): Unit = {
    checkArgument(isDomainId(releaseId) && isReleaseId(releaseId), s"Provided ID '$releaseId' must be a valid release ID")
    val releaseStatus = releaseRepository.getStatus(releaseId)
    checkArgument(releaseStatus != null, s"Provided ID '$releaseId' must exist in the database")
    checkArgument(releaseStatus != ReleaseStatus.TEMPLATE, s"Provided entity '$releaseId' must be a release")
  }

  private def validateTask(taskId: String): Unit = {
    checkArgument(isTaskId(taskId), s"Provided ID '$taskId' must be a valid task ID")
    checkArgument(taskRepository.exists(taskId), s"Provided task ID '$taskId' must exist in the database")
  }
}
