package com.xebialabs.xlrelease.delivery.transition.condition.time

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.delivery.transition.condition.{ConditionChange, DeliveryConditionReaction}
import com.xebialabs.xlrelease.domain.delivery.Condition
import com.xebialabs.xlrelease.domain.delivery.conditions.TimeCondition
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.JobKey.jobKey
import org.quartz.TriggerBuilder.newTrigger
import org.quartz._
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

object TimeConditionReaction {
  val triggerType: String = "delivery.TimeCondition"
}

@Component
class TimeConditionReaction @Autowired()(@Qualifier("xlrQuartzScheduler") scheduler: Scheduler)
  extends DeliveryConditionReaction with Logging {

  override def forType: Type = Type.valueOf(classOf[TimeCondition])

  override def onCreate(change: ConditionChange): Unit = if (change.transition.isAutomated) {
    schedule(change.condition.asInstanceOf[TimeCondition])
  }

  override def onUpdate(change: ConditionChange): Unit = {
    val condition = change.condition.asInstanceOf[TimeCondition]
    if (change.transition.isAutomated) {
      schedule(condition)
    } else {
      unschedule(condition)
    }
  }

  override def onDelete(change: ConditionChange): Unit = unschedule(change.condition.asInstanceOf[TimeCondition])

  //

  protected def schedule(condition: TimeCondition): Unit = if (condition.getFixedDate == null) {
    logger.debug(s"Unable to schedule un-configured time condition '${condition.getId}'")
  } else {
    logger.debug(s"Scheduling time condition '${condition.getId}' to fire on '${condition.getFixedDate}'")
    val jobClass = classOf[TimeConditionQuartzJob]

    val job: JobDetail = newJob(jobClass)
      .withDescription(condition.getId)
      .withIdentity(getJobKey(condition))
      .build()

    val trigger = newTrigger()
      .withIdentity(condition.getId, TimeConditionReaction.triggerType)
      .withDescription(condition.getId)
      .startAt(condition.getFixedDate)
      .forJob(job)
      .build()

    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.rescheduleJob(trigger.getKey, trigger)
    } else {
      scheduler.scheduleJob(job, Set(trigger).asJava, true)
    }
  }

  protected def unschedule(condition: TimeCondition): Unit = {
    logger.debug(s"Un-scheduling time condition '${condition.getId}'")
    scheduler.deleteJob(getJobKey(condition))
  }

  protected def getJobKey(condition: Condition): JobKey = {
    jobKey(condition.getId, TimeConditionReaction.triggerType)
  }

}
