package com.xebialabs.xlrelease.api.v1.impl;

import java.util.Collections;
import java.util.List;
import javax.annotation.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.api.v1.DeliveryApi;
import com.xebialabs.xlrelease.api.v1.forms.CompleteTransition;
import com.xebialabs.xlrelease.api.v1.forms.DeliveryFilters;
import com.xebialabs.xlrelease.api.v1.forms.DeliveryOrderMode;
import com.xebialabs.xlrelease.api.v1.views.DeliveryFlowReleaseInfo;
import com.xebialabs.xlrelease.api.v1.views.DeliveryTimeline;
import com.xebialabs.xlrelease.delivery.actors.DeliveryActorService;
import com.xebialabs.xlrelease.delivery.security.DeliveryPermissionChecker;
import com.xebialabs.xlrelease.delivery.service.DeliveryExecutionService;
import com.xebialabs.xlrelease.delivery.service.DeliveryService;
import com.xebialabs.xlrelease.domain.delivery.*;
import com.xebialabs.xlrelease.exception.LogFriendlyConcurrentModificationException;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.Page;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.domain.utils.DeliveryUtils.deliveryIdFrom;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isStageId;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;

@Controller
public class DeliveryApiImpl implements DeliveryApi {
    private final DeliveryActorService deliveryActorService;
    private final DeliveryService deliveryService;
    private final DeliveryExecutionService deliveryExecutionService;
    private final DeliveryPermissionChecker permissions;

    @Autowired
    public DeliveryApiImpl(DeliveryActorService deliveryActorService,
                           DeliveryService deliveryService,
                           DeliveryExecutionService deliveryExecutionService,
                           DeliveryPermissionChecker permissions) {
        this.deliveryActorService = deliveryActorService;
        this.deliveryService = deliveryService;
        this.deliveryExecutionService = deliveryExecutionService;
        this.permissions = permissions;
    }

    @Timed
    @Override
    public Delivery getDelivery(String deliveryId) {
        permissions.checkViewDelivery(deliveryId);
        return deliveryService.getDelivery(deliveryId);
    }

    @Timed
    @Override
    public Delivery updateDelivery(String deliveryId, Delivery delivery) {
        permissions.checkEditDelivery(deliveryId);
        permissions.checkEditDelivery(delivery);

        delivery.setId(deliveryId);
        // Users can't change the status of release delivery
        Delivery existingDelivery = deliveryService.getDelivery(deliveryId);
        delivery.setStatus(existingDelivery.getStatus());

        return deliveryActorService.updateDelivery(delivery);
    }

    @Timed
    @Override
    public Delivery updateDelivery(Delivery delivery) {
        return updateDelivery(delivery.getId(), delivery);
    }

    @Timed
    @Override
    public void deleteDelivery(String deliveryId) {
        permissions.checkEditDelivery(deliveryId);
        deliveryActorService.deleteDelivery(deliveryId);
    }

    @Timed
    @Override
    public List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters,
                                           Long page,
                                           Long resultsPerPage,
                                           DeliveryOrderMode orderBy) {
        checkArgument(resultsPerPage <= 100, "Number of results per page cannot be more than 100");

        if (null == deliveryFilters) {
            deliveryFilters = new DeliveryFilters();
        }
        return deliveryService.search(deliveryFilters, Page.parse(ofNullable(page), ofNullable(resultsPerPage), empty()), orderBy);
    }

    @Timed
    @Override
    public List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters, @Nullable DeliveryOrderMode orderBy) {
        return searchDeliveries(deliveryFilters, 0L, 100L, orderBy);
    }

    @Timed
    @Override
    public List<Delivery> searchDeliveries(DeliveryFilters deliveryFilters) {
        return searchDeliveries(deliveryFilters, null);
    }

    @Timed
    @Override
    public DeliveryTimeline getDeliveryTimeline(String deliveryId) {
        permissions.checkViewDelivery(deliveryId);
        return deliveryService.getTimeline(deliveryId, DateTime.now());
    }

    @Timed
    @Override
    public List<DeliveryFlowReleaseInfo> getReleases(String deliveryId) {
        permissions.checkViewDelivery(deliveryId);
        return deliveryService.getReleases(deliveryId);
    }

    // SUBSCRIBERS

    @Timed
    @Override
    public SubscriptionResult registerSubscriber(String deliveryId, Subscriber subscriber) {
        permissions.checkViewDelivery(deliveryId);
        return deliveryActorService.registerSubscriber(deliveryId, subscriber);
    }

    // TRACKED ITEM ENDPOINTS

    @Timed
    @Override
    public TrackedItem createTrackedItem(String deliveryId, TrackedItem item) {
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        return deliveryActorService.createTrackedItem(deliveryId, item);
    }

    @Timed
    @Override
    public List<TrackedItem> getTrackedItems(String deliveryId) {
        permissions.checkViewDelivery(deliveryId);
        return deliveryExecutionService.getTrackedItems(deliveryId);
    }

    @Timed
    @Override
    public TrackedItem updateTrackedItem(String itemId, TrackedItem item) {
        item.setId(itemId);
        String deliveryId = deliveryIdFrom(itemId);
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        return deliveryActorService.updateTrackedItem(deliveryId, item);
    }

    @Timed
    @Override
    public TrackedItem updateTrackedItem(TrackedItem item) {
        return updateTrackedItem(item.getId(), item);
    }

    @Timed
    @Override
    public void deleteTrackedItem(String itemId) {
        String deliveryId = deliveryIdFrom(itemId);
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        deliveryActorService.deleteTrackedItem(deliveryId, itemId);
    }

    @Timed
    @Override
    public void descopeTrackedItem(String itemId) {
        String deliveryId = deliveryIdFrom(itemId);
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        deliveryActorService.descopeTrackedItem(deliveryId, itemId);
    }

    @Timed
    @Override
    public void rescopeTrackedItem(String itemId) {
        String deliveryId = deliveryIdFrom(itemId);
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        deliveryActorService.rescopeTrackedItem(deliveryId, itemId);
    }

    @Timed
    @Override
    public void completeTrackedItem(String stageId, String itemId) {
        String deliveryId = deliveryIdFrom(stageId);
        markTrackedItemsInStage(deliveryId, stageId, Collections.singletonList(itemId), TrackedItemStatus.READY, false, null);
    }

    @Timed
    @Override
    public void skipTrackedItem(String stageId, String itemId) {
        String deliveryId = deliveryIdFrom(stageId);
        permissions.checkEditTrackedItemOnStage(stageId);
        deliveryActorService.skipTrackedItem(deliveryId, stageId, itemId);
    }

    @Timed
    @Override
    public void resetTrackedItem(String stageId, String itemId) {
        String deliveryId = deliveryIdFrom(stageId);
        permissions.checkEditTrackedItemOnStage(stageId);
        deliveryActorService.resetTrackedItem(deliveryId, stageId, itemId);
    }

    @Timed
    @Override
    public void registerTrackedItems(String deliveryId, List<String> itemIdOrTitles, String fromReleaseId) {
        permissions.checkEditTrackedItemOnDelivery(deliveryId);
        deliveryActorService.registerTrackedItems(deliveryId, itemIdOrTitles, fromReleaseId);
    }

    @Timed
    @Override
    public List<TrackedItem> markTrackedItemsInStage(String deliveryId, String stageIdOrTitle, List<String> itemIdOrTitles, TrackedItemStatus status, boolean precedingStages, String releaseId) {
        if (isStageId(stageIdOrTitle)) {
            permissions.checkEditTrackedItemOnStage(stageIdOrTitle);
        } else {
            permissions.checkEditTrackedItemOnDelivery(deliveryId);
        }
        return deliveryActorService.markTrackedItemsInStage(deliveryId, stageIdOrTitle, itemIdOrTitles, status, precedingStages, releaseId);
    }

    // STAGE ENDPOINTS

    @Timed
    @Override
    public void completeStage(String stageId) {
        permissions.checkEditDeliveryStage(stageId);
        deliveryActorService.completeStage(deliveryIdFrom(stageId), stageId);
    }

    @Timed
    @Override
    public void reopenStage(String stageId) {
        permissions.checkEditDeliveryStage(stageId);
        deliveryActorService.reopenStage(deliveryIdFrom(stageId), stageId);
    }

    @Timed
    @Override
    public List<Stage> getStages(String deliveryId) {
        permissions.checkViewDeliveryPattern(deliveryId);
        return deliveryService.getDelivery(deliveryId).getStages();
    }

    @Timed
    @Override
    public Stage updateStage(Stage stage) {
        return updateStage(stage.getId(), stage);
    }

    @Timed
    @Override
    public Stage updateStage(String stageId, Stage stage) {
        stage.setId(stageId);
        permissions.checkEditDeliveryStage(stageId);
        try {
            return deliveryActorService.updateStage(deliveryIdFrom(stageId), stage, false);
        } catch (LogFriendlyNotFoundException e) {
            throw new LogFriendlyConcurrentModificationException(e, "%s", e.getMessage());
        }
    }

    // TRANSITION ENDPOINTS

    @Timed
    @Override
    public Transition updateTransition(String transitionId, Transition transition) {
        checkNotNull(transition, "Transition");
        permissions.checkEditDeliveryStage(getParentId(transitionId));
        transition.setId(transitionId);
        transition.checkRestrictions();

        return deliveryActorService.updateTransition(deliveryIdFrom(transitionId), transition);
    }

    @Timed
    @Override
    public Transition updateTransition(Transition transition) {
        return updateTransition(transition.getId(), transition);
    }

    @Timed
    @Override
    public void completeTransition(String transitionId, CompleteTransition completeTransition) {
        permissions.checkEditDeliveryStage(getParentId(transitionId));
        deliveryActorService.completeTransition(deliveryIdFrom(transitionId), transitionId, completeTransition);
    }
}
