package com.xebialabs.xlrelease.delivery.service

import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.xlrelease.delivery.events._
import com.xebialabs.xlrelease.delivery.transition.{TransitionEvaluator, TransitionParams, TransitionResult}
import com.xebialabs.xlrelease.delivery.util.DeliveryObjectFactory
import com.xebialabs.xlrelease.domain.delivery.DeliveryStatus._
import com.xebialabs.xlrelease.domain.delivery.TrackedItemStatus._
import com.xebialabs.xlrelease.domain.delivery._
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.service.CiIdService

import java.util.Date
import scala.collection.mutable
import scala.compat.java8.OptionConverters._
import scala.jdk.CollectionConverters._

object DeliveryStateManager {
  def apply(delivery: Delivery, ciIdService: CiIdService): DeliveryStateManager =
    new DeliveryStateManager(delivery, new DeliveryObjectFactory(ciIdService))
}

class DeliveryStateManager(val delivery: Delivery, factory: DeliveryObjectFactory) {
  private val events = mutable.Buffer.empty[DeliveryEvent]
  private val messages = mutable.Buffer.empty[String]

  def getEvents: List[DeliveryEvent] = events.toList

  def getMessages: List[String] = messages.toList

  def addTrackedItem(item: TrackedItem): TrackedItem = {
    checkDeliveryIsActive()

    val itemId = factory.trackedItemId(delivery.getId)
    item.setId(itemId)
    item.setModifiedDate(new Date())

    delivery.addTrackedItem(item)
    delivery.addReleaseIds(item.getReleaseIds)

    events += ItemCreatedEvent(item, delivery)
    if (!delivery.isTemplate) {

      delivery.getStagesBeforeFirstOpenTransition.forEach { stage =>
        val stageTrackedItem = factory.createStageTrackedItem(stage.getId, itemId)
        stage.getItems.add(stageTrackedItem)
        events += ItemAvailableEvent(item, stage, delivery)
        if (stage.isClosed) {
          stageTrackedItem.setStatus(SKIPPED)
          events += ItemSkippedEvent(item, stage, delivery)
        }
      }
    }

    item
  }

  def updateTrackedItem(item: TrackedItem): TrackedItem = {
    checkDeliveryIsActive()
    val updatedItem = delivery.getItemByIdOrTitle(item.getId)

    val originalItem = cloneCi(updatedItem)

    updatedItem.setTitle(item.getTitle)
    updatedItem.setModifiedDate(new Date())

    events += ItemUpdatedEvent(updatedItem, delivery, originalItem)
    updatedItem
  }

  def deleteTrackedItem(itemId: String): Unit = {
    checkDeliveryIsActive()
    val item = delivery.getItemByIdOrTitle(itemId)

    delivery.removeTrackedItem(itemId)

    if (!delivery.isTemplate) {
      delivery.getStages.forEach(_.removeTrackedItem(itemId))
    }

    events += ItemRemovedEvent(item, delivery)
  }

  def addReleaseToItem(item: TrackedItem, releaseId: String): Unit = {
    checkDeliveryIsActive()

    item.addReleaseId(releaseId)
    delivery.addReleaseId(releaseId)
  }

  def registerSubscriber(subscriber: Subscriber): SubscriptionResult = {
    checkDeliveryIsActive()

    addSubscriber(subscriber)
  }

  // LIFECYCLE METHODS

  def start(): Delivery = {
    delivery.setStatus(IN_PROGRESS)

    if (hasStages) {
      val firstStageGroup = delivery.getStagesBeforeFirstOpenTransition.asScala.toSeq
      startStageGroup(firstStageGroup, delivery.getTrackedItems.asScala.toSeq)
    }

    delivery
  }

  def markTrackedItemsInStage(items: Seq[TrackedItem], stage: Stage, status: TrackedItemStatus,
                              precedingStages: Boolean = false, fromReleaseId: String = null): Seq[TrackedItem] = {
    checkDeliveryIsEditable()
    checkTrue(stage.isOpen,
      "Stage must be open in order to update items in it")

    val descopedItems = items.filter(_.isDescoped)
    checkTrue(descopedItems.isEmpty,
      s"Unable to complete task, the following tracked items are de-scoped:\n * ${descopedItems.map(_.getTitle).mkString("\n * ")}\n")

    if (precedingStages) {
      delivery.getStagesBefore(stage).asScala
        .dropWhile(_.isClosed)
        .foreach { stage =>
          val itemsInStage = items.filter(item => stage.findItemById(item.getId).isPresent)
          markTrackedItemsInStage(itemsInStage, stage, status, precedingStages = false, fromReleaseId)
        }
    }


    items.foreach { item =>
      stage.findItemById(item.getId).asScala.foreach(_.addReleaseId(fromReleaseId))
    }

    val doneItems = stage.getItems.asScala
      .filter(stageItem => stageItem.getStatus == status)
      .map(_.getTrackedItemId).toSet

    val (done, notDone) = items.partition(item => doneItems.contains(item.getId))

    if (done.nonEmpty) {
      messages += s"Tracked items [${done.map(_.getId).mkString(", ")}] are already in status on stage '${stage.getId}'"
    }

    notDone.foreach(markTrackedItemInStage(_, stage, status, fromReleaseId))
    notDone
  }

  def markTrackedItemInStage(item: TrackedItem, stage: Stage, status: TrackedItemStatus, fromReleaseId: String): Unit = status match {
    case READY => completeTrackedItemInStage(item, stage, fromReleaseId)
    case SKIPPED => skipTrackedItemInStage(item, stage)
    case NOT_READY => resetTrackedItemInStage(item, stage, fromReleaseId)
    case _ => // for future?
  }

  def completeTrackedItemInStage(item: TrackedItem, stage: Stage, fromReleaseId: String = null): Delivery = {
    checkItemIsEditable(item, stage)
    val stageItem = stage.getItemById(item.getId)
    if (stageItem.getStatus != READY) {
      stageItem.setStatus(READY)
      events += ItemCompletedEvent(item, stage, delivery, fromReleaseId)
    }
    delivery
  }

  def completeStage(stage: Stage): Unit = {
    checkDeliveryIsEditable()
    checkTrue(isAllowedToCloseStage(stage),
      "Stages cannot be completed if their preceding stages are not closed")

    val toDescopeAndSkip = stage.getItems.asScala
      .filter(_.getStatus == NOT_READY)
      .map(stageItem => delivery.getItemByIdOrTitle(stageItem.getTrackedItemId)).toSeq
    forceCloseStagesIfNotClosed(Seq(stage), toDescopeAndSkip)
  }

  def reopenStage(stage: Stage): Unit = {
    checkDeliveryIsEditable()
    checkTrue(isAllowedToReopenStage(stage),
      "Stages cannot be reopened if their subsequent stages are closed")

    if (stage.isClosed) {
      stage.setStatus(StageStatus.OPEN)
      events += StageReopenedEvent(stage, delivery)

      // Reinitialize conditions
      Option(stage.getTransition).foreach(_.getAllConditions.forEach(_.reset()))
    }
  }

  def descopeItems(items: Seq[TrackedItem]): Unit = {
    val itemsNotYetDescoped = items.view.filterNot(_.isDescoped).map(item => item.getId -> item).toMap
    itemsNotYetDescoped.valuesIterator.foreach { item =>
      item.setDescoped(true)
      events += ItemDescopedEvent(item, delivery)
    }
  }

  def rescopeItems(items: Seq[TrackedItem]): Unit = {
    val descopedItems = items.view.filter(_.isDescoped).map(item => item.getId -> item).toMap
    descopedItems.valuesIterator.foreach { item =>
      item.setDescoped(false)
      events += ItemRescopedEvent(item, delivery)
    }
  }

  def skipTrackedItemInStage(item: TrackedItem, stage: Stage): Delivery = {
    checkItemIsEditable(item, stage)
    val stageItem = stage.getItemById(item.getId)
    forceSkipTrackedItemInStageIfNotReady(item, stage, stageItem)
    delivery
  }

  def completeDeliveryIfNecessary(): Unit = if (delivery.isAutoComplete && events.exists(item =>
    item.isInstanceOf[ItemCompletedEvent] ||
      item.isInstanceOf[ItemSkippedEvent] ||
      item.isInstanceOf[ItemDescopedEvent])
  ) {
    val allItemsClosedInAllStages = delivery.getTrackedItems.asScala.forall { item =>
      item.isDescoped || delivery.getStages.asScala.forall(stage => stage.findItemById(item.getId).asScala.exists(_.getStatus.isDone))
    }
    if (allItemsClosedInAllStages) {
      forceCloseStagesIfNotClosed(delivery.getStages.asScala.toSeq)
    }
  }

  def resetTrackedItemInStage(item: TrackedItem, stage: Stage, fromReleaseId: String = null): Delivery = {
    checkItemIsEditable(item, stage)
    val stageItem = stage.getItemById(item.getId)

    if (stageItem.getStatus != NOT_READY) {
      stageItem.setStatus(NOT_READY)
      events += ItemResetEvent(item, stage, delivery, fromReleaseId)
      events += ItemAvailableEvent(item, stage, delivery)
    }

    delivery
  }


  def startStageGroup(stages: Seq[Stage], items: Seq[TrackedItem]): Unit = {
    stages.foreach { stage =>
      // eliminate already existing items
      val existingItems = stage.getItems.asScala.map(_.getTrackedItemId).toSet
      val newItems = items.filterNot(item => existingItems(item.getId))

      val stageItems = newItems.map { item =>
        factory.createStageTrackedItem(stage.getId, item.getId)
      }.asJava
      stage.addTrackedItems(stageItems)
      if (existingItems.isEmpty) {
        events += StageStartedEvent(stage, delivery)
      }
      newItems.foreach(events += ItemAvailableEvent(_, stage, delivery))

      if (stage.isClosed) {
        stageItems.forEach(stageItem => stageItem.setStatus(SKIPPED))
        newItems.foreach(events += ItemSkippedEvent(_, stage, delivery))
      }
    }
  }

  def handleTransition(transition: Transition, params: TransitionParams): Unit = {
    TransitionEvaluator.evaluate(params)
      .foreach(result => handleTransition(transition, params, result))
  }

  private def handleTransition(transition: Transition, params: TransitionParams, transitionResult: TransitionResult): Unit = {
    events += TransitionExecutedEvent(transition, delivery, params)

    val stage = delivery.getStageByTransition(transition)
    transitionResult.toTransition.toList.foreach { item =>
      events += ItemTransitionApprovedEvent(item, stage.findItemById(item.getId).get, transition, stage, delivery)
    }

    if (transitionResult.closeStages) {
      val stagesToClose = delivery.getStagesBefore(params.transitionStage).asScala
        .filter(stage => stage.isOpen) :+ params.transitionStage
      forceCloseStagesIfNotClosed(stagesToClose.toSeq, transitionResult.toDescope.toList)
    }

    val nextStageGroup = delivery.getStageGroupAfterTransition(transition).asScala.toSeq
    startStageGroup(nextStageGroup, transitionResult.toTransition.toList)
  }

  private def forceCloseStagesIfNotClosed(stages: Seq[Stage], itemsToDescope: Seq[TrackedItem] = Seq.empty): Unit = {
    stages.foreach { stage =>
      // Only completed and skipped items are allowed in a closed stage
      itemsToDescope.foreach { item =>
        stage.findItemById(item.getId).ifPresent { stageItem =>
          forceSkipTrackedItemInStageIfNotReady(item, stage, stageItem)
        }
      }

      // Deactivate the automatic conditions
      Option(stage.getTransition).foreach(_.setAutomated(false))

      if (stage.isOpen) {
        stage.setStatus(StageStatus.CLOSED)
        events += StageCompletedEvent(stage, delivery)
      }

      if (delivery.isLastStage(stage)) {
        delivery.setStatus(DeliveryStatus.COMPLETED)
        events += DeliveryCompletedEvent(delivery)
      }
    }
    descopeItems(itemsToDescope)
  }

  private def forceSkipTrackedItemInStageIfNotReady(item: TrackedItem, stage: Stage, stageTrackedItem: StageTrackedItem): Unit = {
    if (stageTrackedItem.getStatus == TrackedItemStatus.NOT_READY) {
      stageTrackedItem.setStatus(SKIPPED)
      events += ItemSkippedEvent(item, stage, delivery)
    }
  }

  private def hasStages: Boolean = delivery.getStages != null && !delivery.getStages.isEmpty

  private def isAllowedToCloseStage(stage: Stage) =
    delivery.findPreviousStage(stage).asScala.forall(_.getStatus == StageStatus.CLOSED)

  private def isAllowedToReopenStage(stage: Stage) =
    delivery.findNextStage(stage).asScala.forall(_.getStatus != StageStatus.CLOSED)

  private def checkDeliveryIsActive(): Unit = {
    checkTrue(!DeliveryStatus.INACTIVE_STATUSES.contains(delivery.getStatus),
      "Can't modify ABORTED or COMPLETED release delivery")
  }

  private def addSubscriber(subscriber: Subscriber): SubscriptionResult = {
    subscriber.validate(delivery)
    val result = subscriber.evaluate(delivery)
    val subscriberIndex = delivery.getSubscribers.asScala.indexWhere(_.isEqual(subscriber))

    import com.xebialabs.xlrelease.domain.delivery.SubscriptionStatus._
    result match {
      case SubscriptionResult(COMPLETED, _) if subscriberIndex != -1 =>
        delivery.getSubscribers.remove(subscriberIndex)
      case SubscriptionResult(FAILED, _) | SubscriptionResult(IN_PROGRESS, _) if subscriberIndex != -1 =>
        delivery.getSubscribers.set(subscriberIndex, subscriber)
      case SubscriptionResult(FAILED, _) | SubscriptionResult(IN_PROGRESS, _) =>
        delivery.addSubscriber(subscriber)
      case _ => // ignore all others
    }
    result
  }

  private def checkItemIsEditable(item: TrackedItem, stage: Stage): Unit = {
    checkDeliveryIsEditable()
    checkTrue(stage.isOpen,
      "Stage must be open in order to update items in it")
    checkTrue(!item.isDescoped,
      "Cannot change status of descoped item")
  }

  private def checkDeliveryIsEditable(): Unit = {
    checkTrue(delivery.getStatus == IN_PROGRESS,
      "Delivery must be active in order to change state")
  }
}
