package com.xebialabs.xlrelease.delivery.service

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.api.internal.{CachedInternalMetadataDecorator, DecoratorCache, EffectiveSecurity, InternalMetadata}
import com.xebialabs.xlrelease.api.v1.views.DeliveryDetails
import com.xebialabs.xlrelease.delivery.repository.DeliveryRepository
import com.xebialabs.xlrelease.delivery.service.DeliveryDetailsDecorator.DELIVERY_DETAILS
import com.xebialabs.xlrelease.domain.delivery.Delivery
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.FolderRepository
import com.xebialabs.xlrelease.security.XLReleasePermissions.{VIEW_DELIVERY_PATTERN, VIEW_FOLDER}
import grizzled.slf4j.Logging
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object DeliveryDetailsDecorator {
  val DELIVERY_DETAILS = "details"
}

case class DeliveryDetailsDecoratorCache(
                                          patterns: mutable.Map[String, Try[Delivery]],
                                          folders: mutable.Map[String, Try[Option[Folder]]])

@Component
class DeliveryDetailsDecorator @Lazy()(deliveryRepository: DeliveryRepository, folderRepository: FolderRepository)
  extends CachedInternalMetadataDecorator[Delivery, DeliveryDetailsDecoratorCache] with Logging {

  override val name: String = DELIVERY_DETAILS

  override def isApplicableTo(ci: ConfigurationItem): Boolean = ci.isInstanceOf[Delivery]

  override def decorate(cis: util.Collection[Delivery], cache: DecoratorCache[DeliveryDetailsDecoratorCache]): Unit = {
    val cached: DeliveryDetailsDecoratorCache = cache.computeIfAbsent(() => DeliveryDetailsDecoratorCache(mutable.Map[String, Try[Delivery]](), mutable.Map[String, Try[Option[Folder]]]()))
    cis.asScala.foreach { del =>
      val metadata = new DeliveryDetails
      resolveOriginPattern(del, cached) match {
        case Success(pattern) => metadata.setPatternDetails(pattern.getTitle,
          hasPermission(pattern.get$metadata().get("security"), VIEW_DELIVERY_PATTERN.getPermissionName))
        case Failure(e) => logger.debug(s"Cannot find pattern with id ${del.getOriginPatternId}", e)
      }
      if (!cached.folders.contains(del.getFolderId)) {
        cached.folders.put(del.getFolderId, Try(folderRepository.findById(del.getFolderId, 1)))
      }
      cached.folders(del.getFolderId) match {
        case Success(maybeTitle) => maybeTitle match {
          case Some(folder) => metadata.setFolderDetails(folder.getTitle,
            hasPermission(folder.get$metadata().get("security"), VIEW_FOLDER.getPermissionName))
          case None => metadata.setFolderDetails("Folder is not defined", false)
        }
        case Failure(e) => logger.debug(s"Cannot find folder with id ${del.getFolderId}", e)
      }
      del.get$metadata().put(DELIVERY_DETAILS, metadata)
    }

  }

  // had a Stackoverflow with misconfigured delivery here (getByIdOrTitle -> this decorator -> getByIdOrTitle), this is just for the `just in case`
  private def resolveOriginPattern(delivery: Delivery, cached: DeliveryDetailsDecoratorCache): Try[Delivery] = {
    if (delivery.getOriginPatternId == null || delivery.getId == delivery.getOriginPatternId) {
      Try(delivery)
    } else {
      if (!cached.patterns.contains(delivery.getOriginPatternId)) {
        cached.patterns.put(delivery.getOriginPatternId, Try(deliveryRepository.getByIdOrTitle(delivery.getOriginPatternId)))
      }
      cached.patterns(delivery.getOriginPatternId)
    }
  }

  private def hasPermission(security: InternalMetadata, permissionName: String): Boolean = {
    val maybeSecurity = Option(security.asInstanceOf[EffectiveSecurity])
    (for {
      security <- maybeSecurity
      permissions <- Option(security.getPermissions)
    } yield
      permissions.contains(permissionName)
      ) getOrElse false
  }
}
