import { SagaIterator } from 'redux-saga';
import { call, put, select } from 'redux-saga/effects';
import getAngularService from '../../../../../../../../../core/xlr-ui/app/features/common/services/angular-accessor';
import { navigation } from '../../../../../../../../../core/xlr-ui/app/features/main-navigation/ducks/navigation.reducer';

import { Delivery } from '../../Tasks/js/components/types';
import { DeliverySidebarProvider, selectDelivery } from './delivery-sidebar-provider';

const { setSidebarMenu, setBackItem, setIsInTemplateOrRelease } = navigation.actions;

describe('DeliverySidebarProvider', () => {
    const deliverySidebarProvider = new DeliverySidebarProvider();

    const path = '/deliveries/Deliveries/Deliveryb54aafcfc134439087a2189553d775a7';

    describe('isSupported', () => {
        it('should be true for delivery patterns url', () => {
            expect(deliverySidebarProvider.isSupported(path)).toBe(true);
        });
    });

    describe('provide', () => {
        const delivery: Delivery = {
            id: 'Deliveryb54aafcfc134439087a2189553d775a7',
            stages: [],
            title: 'My delivery pattern',
            folderId: 'Folder123',
        };

        const uiExtensionsService = {
            getMenuItems: jest.fn(),
        };

        const deliveryItems = [
            {
                label: 'Delivery flow',
                pathSuffix: '',
                properties: {},
                uri: 'include/deliveries-details/components/delivery-flow/index.html',
                weight: 10,
                permitted: true,
            },
            {
                label: 'Timeline',
                pathSuffix: 'timeline',
                properties: {},
                uri: 'include/deliveries-details/components/timeline/index.html',
                weight: 20,
                permitted: true,
            },
            {
                label: 'Properties',
                pathSuffix: 'properties',
                properties: {},
                uri: 'include/deliveries-details/components/properties/index.html',
                weight: 30,
                permitted: true,
            },
            {
                label: 'Activity logs',
                pathSuffix: 'logs',
                properties: {},
                uri: 'managed-by-custom-state',
                weight: 40,
                permitted: true,
            },
        ];

        const verifyBegin = (gen: SagaIterator) => {
            expect(gen.next().value).toStrictEqual(select(selectDelivery));
            expect(gen.next(delivery).value).toStrictEqual(call(getAngularService, 'UiExtensionsService'));
            expect(gen.next(uiExtensionsService).value).toStrictEqual(
                call([uiExtensionsService, uiExtensionsService.getMenuItems], 'xlrelease.DeliveryPageMenu', delivery),
            );
        };

        const verifyEnd = (gen: SagaIterator) => {
            expect(gen.next(deliveryItems).value).toStrictEqual(put(setIsInTemplateOrRelease(true)));
            expect(gen.next().value).toStrictEqual(
                put(
                    setBackItem({
                        pathSuffix: 'folders/Folder123/deliveries',
                        text: 'Deliveries',
                        title: 'deliveries',
                    }),
                ),
            );
            expect(gen.next().done).toBe(true);
        };

        it('should call proper sagas for default page', () => {
            const gen: SagaIterator = deliverySidebarProvider.provide(path);
            verifyBegin(gen);
            expect(gen.next(deliveryItems).value).toStrictEqual(
                put(
                    setSidebarMenu([
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7',
                            selected: true,
                            startIconId: 'file',
                            text: 'Delivery flow',
                            weight: 10,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/timeline',
                            selected: false,
                            startIconId: 'file',
                            text: 'Timeline',
                            weight: 20,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/properties',
                            selected: false,
                            startIconId: 'file',
                            text: 'Properties',
                            weight: 30,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/logs',
                            selected: false,
                            startIconId: 'file',
                            text: 'Activity logs',
                            weight: 40,
                        },
                    ]),
                ),
            );
            verifyEnd(gen);
        });
        it('should call proper sagas for properties page', () => {
            const gen: SagaIterator = deliverySidebarProvider.provide(`${path}/properties`);
            verifyBegin(gen);
            expect(gen.next(deliveryItems).value).toStrictEqual(
                put(
                    setSidebarMenu([
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7',
                            selected: false,
                            startIconId: 'file',
                            text: 'Delivery flow',
                            weight: 10,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/timeline',
                            selected: false,
                            startIconId: 'file',
                            text: 'Timeline',
                            weight: 20,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/properties',
                            selected: true,
                            startIconId: 'file',
                            text: 'Properties',
                            weight: 30,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/logs',
                            selected: false,
                            startIconId: 'file',
                            text: 'Activity logs',
                            weight: 40,
                        },
                    ]),
                ),
            );
            verifyEnd(gen);
        });
        it('should call proper sagas for timeline page', () => {
            const gen: SagaIterator = deliverySidebarProvider.provide(`${path}/timeline`);
            verifyBegin(gen);
            expect(gen.next(deliveryItems).value).toStrictEqual(
                put(
                    setSidebarMenu([
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7',
                            selected: false,
                            startIconId: 'file',
                            text: 'Delivery flow',
                            weight: 10,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/timeline',
                            selected: true,
                            startIconId: 'file',
                            text: 'Timeline',
                            weight: 20,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/properties',
                            selected: false,
                            startIconId: 'file',
                            text: 'Properties',
                            weight: 30,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/logs',
                            selected: false,
                            startIconId: 'file',
                            text: 'Activity logs',
                            weight: 40,
                        },
                    ]),
                ),
            );
            verifyEnd(gen);
        });
        it('should call proper sagas for activity logs', () => {
            const gen: SagaIterator = deliverySidebarProvider.provide(`${path}/logs`);
            verifyBegin(gen);
            expect(gen.next(deliveryItems).value).toStrictEqual(
                put(
                    setSidebarMenu([
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7',
                            selected: false,
                            startIconId: 'file',
                            text: 'Delivery flow',
                            weight: 10,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/timeline',
                            selected: false,
                            startIconId: 'file',
                            text: 'Timeline',
                            weight: 20,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/properties',
                            selected: false,
                            startIconId: 'file',
                            text: 'Properties',
                            weight: 30,
                        },
                        {
                            pathSuffix: 'deliveries/Deliveryb54aafcfc134439087a2189553d775a7/logs',
                            selected: true,
                            startIconId: 'file',
                            text: 'Activity logs',
                            weight: 40,
                        },
                    ]),
                ),
            );
            verifyEnd(gen);
        });
    });
});
