import React, { useEffect, useState } from 'react';
import { DotAutoComplete, DotInputText } from '@digital-ai/dot-components';
import { TaskInputCommonProps } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/rails/config/components/types';
import { StringWithVariables } from './string-with-variables.component';
import { Delivery, DeliveryStage } from '../types';
import { getDeliveryTasksState } from '../ducks/delivery-tasks.reducer';
import { useAppSelector } from '../../../../../../../../../../../core/xlr-ui/app/js/hooks';
import { AUTOCOMPLETE_MAX_HEIGHT } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/common/constants';
import { isDeliveryStageId, useCleanValueWhenChanged } from '../helper';
import { IdAutoCompleteOption } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/types';
import { isVariable } from '../../../../../../../../../../../core/xlr-ui/app/features/tasks/components/helper/variables.helper';

export const StageWithVariables = (props: TaskInputCommonProps) => {
    return (
        <StringWithVariables {...props}>
            <StageAutocomplete {...props} />
        </StringWithVariables>
    );
};

const StageAutocomplete = ({ disabled, onChange, property, task }: TaskInputCommonProps) => {
    const { patterns, deliveryOrPattern } = useAppSelector(getDeliveryTasksState);
    const stageId = task.inputProperties?.stage as string;
    const patternId = task.inputProperties?.patternId as string;
    const deliveryId = task.inputProperties?.deliveryId as string;
    const [stages, setStages] = useState<Array<IdAutoCompleteOption>>([]);
    const [autocompleteValue, setAutocompleteValue] = useState<IdAutoCompleteOption | null>(null);
    const isStageId = stageId ? isDeliveryStageId(stageId) : true;

    useCleanValueWhenChanged(onChange, [patternId, deliveryId]);

    useEffect(() => {
        // we have a pattern selected and loaded, give me the stages of the pattern
        if (patternId && patterns) {
            const selectedPattern = patterns.find((p: Delivery) => p.id === patternId);
            if (selectedPattern) {
                setStages(
                    selectedPattern.stages.map((s: DeliveryStage) => ({
                        id: s.id,
                        title: s.title,
                    })),
                );
            }
        }
    }, [patternId, patterns]);

    useEffect(() => {
        // we have a delivery and is not variable, give me the stages from the delivery
        if (deliveryOrPattern && !isVariable(deliveryId)) {
            setStages(
                deliveryOrPattern.stages.map((s: DeliveryStage) => ({
                    id: s.id,
                    title: s.title,
                })),
            );
        }
    }, [deliveryOrPattern, deliveryId]);

    useEffect(() => {
        // we don't have patternId, neither deliveryId, nothing to select from
        if (!patternId && !deliveryId) {
            setStages([]);
        }
    }, [patternId, deliveryId]);

    useEffect(() => {
        // if the task has finished the stages can be coming from anywhere
        if (disabled && patternId && deliveryId && !isVariable(deliveryId)) {
            const selectedPatternStages = patterns.find((p: Delivery) => p.id === patternId)?.stages || [];
            const selectedDeliveryStages = deliveryOrPattern?.stages || [];
            setStages(
                [...selectedDeliveryStages, ...selectedPatternStages].map((s: DeliveryStage) => ({
                    id: s.id,
                    title: s.title,
                })),
            );
        }
    }, [disabled, deliveryOrPattern, patterns]);

    useEffect(() => {
        setAutocompleteValue(stages.find((o) => o.id === stageId) || null);
    }, [stageId, stages]);

    const onHandleChange = (v: IdAutoCompleteOption) => {
        onChange(v?.id);
    };

    return (
        <div style={{ width: '100%' }}>
            {isStageId && (
                <DotAutoComplete
                    group={false}
                    helperText={property.description}
                    inputId={property.name}
                    key={autocompleteValue?.title}
                    label={property.label}
                    maxHeight={AUTOCOMPLETE_MAX_HEIGHT}
                    multiple={false}
                    onChange={(_e, value) => onHandleChange(value as IdAutoCompleteOption)}
                    options={stages}
                    readOnly={disabled}
                    required={property.required}
                    value={autocompleteValue}
                />
            )}
            {!isStageId && <DotInputText id={property.name} name={property.name} readOnly={disabled} value={stageId} />}
        </div>
    );
};
