package com.xebialabs.xlrelease.delivery.transition

import com.xebialabs.xlrelease.domain.delivery.{Delivery, Stage, TrackedItem, TrackedItemStatus}

import scala.compat.java8.OptionConverters._
import scala.jdk.CollectionConverters._

sealed trait TransitionTrigger

case class UserTrigger(username: String) extends TransitionTrigger

case class ConditionTrigger(conditionId: String, conditionTitle: String) extends TransitionTrigger

case class TransitionParams(delivery: Delivery, transitionStage: Stage,
                            transitionItems: Set[String], closeStages: Boolean,
                            trigger: TransitionTrigger)

case class TransitionResult(toTransition: Set[TrackedItem] = Set.empty, toDescope: Set[TrackedItem] = Set.empty, closeStages: Boolean = true) {
  def merge(other: TransitionResult): TransitionResult =
    this.copy(this.toTransition ++ other.toTransition, this.toDescope ++ other.toDescope, other.closeStages)
}

object TransitionEvaluator {
  def evaluate(params: TransitionParams): Option[TransitionResult] = {
    if (params.transitionStage.isOpen) {
      val transitionResult = {
        val pendingParams = if (params.closeStages) {
          val delivery = params.delivery
          delivery.getStagesBefore(params.transitionStage).asScala
            .filter(stage => stage.isOpen && stage.getTransition != null)
            .map(TransitionParams(delivery, _, Set.empty, closeStages = true, params.trigger))
        } else {
          Seq.empty
        }

        (pendingParams :+ params)
          .foldLeft(TransitionResult()) { case (result, currentTrigger) =>
            val (toTransition, toDescope) = calculateTransition(currentTrigger)
            result.merge(TransitionResult(toTransition.toSet, toDescope.toSet, params.closeStages))
          }
      }

      Some(transitionResult)
    } else {
      None
    }
  }

  private def calculateTransition(params: TransitionParams): (Seq[TrackedItem], Seq[TrackedItem]) = {
    // we only care about transition's stage's items
    val stageTrackedItems = Map(params.transitionStage.getItems.asScala.map(i => (i.getTrackedItemId, i)).toSeq: _*)

    // Descope items if:
    // the items are in the stage and (are not ready or are not selected) and closeStages is selected
    // Transition items if:
    // the items are in the stage and are done (ready or skipped)
    val (toTransition, toDescope) = params.delivery.getTrackedItems.asScala
      .foldRight((Seq.empty[TrackedItem], Seq.empty[TrackedItem])) { case (item, (transitionItems, descopeItems)) =>
        stageTrackedItems.get(item.getId).map { stageTrackedItem =>
          if (stageTrackedItem.getStatus.isDone) {
            if (params.transitionItems.contains(item.getId)) {
              (item +: transitionItems) -> descopeItems
            } else {
              params.delivery.findNextStage(params.transitionStage).asScala match {
                case Some(nextStage) if !nextStage.findItemById(item.getId).isPresent =>
                  transitionItems -> (item +: descopeItems)
                case _ => transitionItems -> descopeItems
              }
            }
          } else if (stageTrackedItem.getStatus == TrackedItemStatus.NOT_READY) {
            transitionItems -> (item +: descopeItems)
          } else {
            transitionItems -> descopeItems
          }
        }.getOrElse(transitionItems -> descopeItems)
      }
    (toTransition, toDescope)
  }
}
