import React, { useEffect, useState } from 'react';
import { DotAutoComplete, AutoCompleteOption } from '@digital-ai/dot-components';
import { Task, TaskInputPropertyValue } from '../../../../../../../../../../core/xlr-ui/app/types';
import { SyntheticPropertyDefinition } from '../../../../../../../../../../core/xlr-ui/app/types/synthetic';
import { searchDeliveryPatterns, fetchDeliveryPattern } from '../../../delivery-patterns/delivery-pattern.service';
import IdsFactory from '../../../../../../../../../../core/xlr-ui/app/js/util/ids';

export interface DeliverySelectorProps {
    expanded: boolean;
    onChange: (value: TaskInputPropertyValue) => void;
    property: SyntheticPropertyDefinition;
    task: Task;
}

interface Delivery {
    id: string;
    title: string;
}

interface AutoCompleteOptionWithId extends AutoCompleteOption {
    id: string;
}

const Ids = IdsFactory();

export const DeliverySelector = ({ onChange, property, task }: DeliverySelectorProps) => {
    const [deliveries, setDeliveries] = useState<Array<Delivery>>([]);
    const [selectedDelivery, setSelectedDelivery] = useState<AutoCompleteOptionWithId | null>(null);
    const [hasPatternViewPermission, setHasPatternViewPermission] = useState(true);
    const [isDeliveryFound, setDeliveryFound] = useState(true);
    const [description, setDescription] = useState(property.description);

    useEffect(() => {
        // this should go through sagas...
        searchDeliveryPatterns({ title: '' }).then((response: Array<Delivery>) => setDeliveries(response));
    }, [property.fqn]);

    useEffect(() => {
        const patternId = task.inputProperties?.[property.name] as string;
        if (patternId) {
            fetchDeliveryPattern(patternId).then((response: Delivery) => {
                setDeliveryFound(true);
                if (response.id === 'forbiddenFolderNotNull' && response.title === '') {
                    setHasPatternViewPermission(false);
                    setDescription('You do not have permissions to view the selected pattern');
                    const pId = Ids.getName(Ids.toDomainId(patternId));
                    const patternWithId: AutoCompleteOptionWithId = { id: patternId, title: pId };
                    setSelectedDelivery(patternWithId);
                } else if (response.id === null && response.title === '') {
                    setDeliveryFound(false);
                    setSelectedDelivery({ id: patternId, title: patternId });
                    setDescription('Currently selected delivery was deleted');
                } else {
                    setHasPatternViewPermission(true);
                    const foundDeliveryOnOptions = deliveriesOptions.find((d) => d.id === patternId) || null;
                    setSelectedDelivery(foundDeliveryOnOptions);
                }
            });
        } else {
            setSelectedDelivery(null);
        }
    }, [deliveries, task.inputProperties?.[property.name]]);

    const handleDeliveryChange = (value?: AutoCompleteOptionWithId) => {
        onChange(value ? value.id : null);
    };

    const deliveriesOptions: Array<AutoCompleteOptionWithId> = deliveries.map((d) => ({ id: d.id, title: d.title }));

    return (
        <div className="task-input-property">
            <DotAutoComplete
                error={!hasPatternViewPermission || !isDeliveryFound}
                freesolo={false}
                helperText={description}
                inputId={property.fqn}
                label={property.label}
                multiple={false}
                onChange={(_e, v) => handleDeliveryChange(v as AutoCompleteOptionWithId)}
                options={deliveriesOptions}
                value={selectedDelivery}
            />
        </div>
    );
};
