package com.xebialabs.xlrelease.quartz.release.scheduler

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.FeatureService
import grizzled.slf4j.Logging
import org.quartz.{JobDetail, Trigger}
import org.springframework.context.event.EventListener

@IsTransactional
trait QuartzScheduledService extends FeatureService with Logging {

  private var running: Boolean = false

  def serviceName(): String

  def releaseSchedulerService: ReleaseSchedulerService

  def getJobDetail: JobDetail

  def getTrigger: Trigger

  def isRunning: Boolean = running

  @EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      try {
        // Some of the services actually override isRunning method which then
        // uses the UI managed setting... soooo you still need to check if this should be running on startup.
        if (isRunning) {
          logger.debug(s"Scheduling ${serviceName()} service")
          releaseSchedulerService.schedule(getJobDetail, getTrigger)
          running = true
        } else {
          logger.debug(s"Removing ${serviceName()} service")
          releaseSchedulerService.unschedule(getJobDetail.getKey)
          running = false
        }
      } catch {
        case e: Exception => logger.error(s"Error while starting ${serviceName()} service", e)
      }
    }
  }

  def stop(): Unit = {
    if (!isRunning) {
      releaseSchedulerService.resume(getJobDetail.getKey)
      logger.debug(s"Started ${serviceName()} service")
    }
    running = true
  }

  def start(): Unit = {
    if (isRunning) {
      releaseSchedulerService.pause(getJobDetail.getKey)
      logger.debug(s"Stopped ${serviceName()} service")
    }
    running = false
  }
}
