package com.xebialabs.xlrelease.quartz.release.scheduler

import grizzled.slf4j.Logging
import org.quartz.{JobDetail, JobKey, ObjectAlreadyExistsException, Scheduler, Trigger}
import org.springframework.scheduling.quartz.SchedulerFactoryBean

class ReleaseSchedulerService(schedulerFactoryBean: SchedulerFactoryBean) extends Logging {

  def getScheduler: Scheduler = schedulerFactoryBean.getScheduler

  def schedule(job: JobDetail, trigger: Trigger): Unit = {
    if (isJobValid(job) && isTriggerValid(trigger)) {
      val triggerKey = trigger.getKey
      val triggerExists = getScheduler.checkExists(triggerKey)
      if (triggerExists) {
        getScheduler.addJob(job, true)
        try {
          getScheduler.rescheduleJob(triggerKey, trigger)
        } catch {
          case ex: ObjectAlreadyExistsException =>
            logger.debug(s"Unexpectedly encountered existing trigger on rescheduling, " +
              s"assumably due to cluster race condition: ${ex.getMessage} - can safely be ignored")
        }
      } else {
        try {
          getScheduler.scheduleJob(job, trigger)
        } catch {
          case ex: ObjectAlreadyExistsException =>
            logger.debug(s"Unexpectedly encountered existing trigger on job scheduling, " +
              s"assumably due to cluster race condition: ${ex.getMessage} - can safely be ignored")
            getScheduler.rescheduleJob(triggerKey, trigger)
        }
      }
    }
  }

  def unschedule(jobKey: JobKey): Boolean = {
    val jobExists = getScheduler.checkExists(jobKey)
    if (jobExists) {
      getScheduler.deleteJob(jobKey)
    } else {
      false
    }
  }

  def resume(jobKey: JobKey): Unit = {
    getScheduler.resumeJob(jobKey)
  }

  def pause(jobKey: JobKey): Unit = {
    getScheduler.pauseJob(jobKey)
  }

  private def isJobValid(job: JobDetail): Boolean = {
    if (null != job && null != job.getKey) {
      true
    } else {
      false
    }
  }

  private def isTriggerValid(trigger: Trigger): Boolean = {
    if (null != trigger && null != trigger.getKey) {
      true
    } else {
      false
    }
  }
}
