package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.quartz.config.QuartzConfiguration.QUARTZ_SCHEDULER_NAME
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.quartz.release.scheduler.ReleaseSchedulerService
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{JobDetail, Trigger}
import org.springframework.context.event
import org.springframework.stereotype.Component

@Component
class StuckGateDetectorScheduler(xlrConfig: XlrConfig,
                                 releaseSchedulerService: ReleaseSchedulerService) extends Logging {

  val JOB_NAME = "STUCK_GATE_DETECTOR_JOB"
  val GROUP_NAME = "STUCK_GATE_DETECTOR"

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    if (null != event && null != event.schedulerName && event.schedulerName == QUARTZ_SCHEDULER_NAME) {
      try {
        val job: JobDetail = newJob(classOf[StuckGateDetectorSchedulerJob])
          .withDescription(JOB_NAME)
          .withIdentity(JOB_NAME, GROUP_NAME)
          .storeDurably(true)
          .build()
        val trigger: Trigger = newTrigger()
          .withIdentity(JOB_NAME, GROUP_NAME)
          .withDescription(JOB_NAME)
          .withSchedule(simpleSchedule()
            .withIntervalInMilliseconds(xlrConfig.features.gateActor.unstuckInterval.toMillis)
            .repeatForever())
          .build()
        if (xlrConfig.features.gateActor.enabled) {
          logger.debug("Scheduling stuck gate detector")
          releaseSchedulerService.schedule(job, trigger)
        } else {
          releaseSchedulerService.unschedule(job.getKey)
        }
      } catch {
        case e: Exception => logger.error(s"Error while starting ${classOf[StuckGateDetectorScheduler].getName} service", e)
      }
    }
  }
}
