package com.xebialabs.xlrelease.quartz.release.scheduler

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.quartz.events.SchedulerStartedEvent
import com.xebialabs.xlrelease.service.FeatureService
import com.xebialabs.xlrelease.upgrade.common.ClassNameToString
import grizzled.slf4j.Logging
import org.quartz.JobBuilder.newJob
import org.quartz.SimpleScheduleBuilder.simpleSchedule
import org.quartz.TriggerBuilder.newTrigger
import org.quartz.{Job, JobDetail, Scheduler, SimpleTrigger}
import org.springframework.context.event

import scala.concurrent.duration.Duration

@IsTransactional
abstract class BaseNotificationService(serviceName: String, jobClazz: Class[_ <: Job], defaultTriggerInterval: Duration, notificationEnabled: Boolean)
  extends ClassNameToString with Logging with FeatureService {

  import scala.jdk.CollectionConverters._

  private var running: Boolean = false
  private val jobName: String = s"$serviceName JOB".toUpperCase.split(' ').mkString("_")
  private val groupName: String = serviceName.toUpperCase.split(' ').mkString("_")

  def triggerInterval: Duration = defaultTriggerInterval

  def scheduler: Scheduler

  lazy val job: JobDetail = newJob(jobClazz)
    .withDescription(s"$serviceName job")
    .withIdentity(jobName, groupName)
    .build()

  lazy val trigger: SimpleTrigger = newTrigger()
    .withIdentity(jobName, groupName)
    .withDescription(s"Trigger $serviceName jobs")
    .withSchedule(simpleSchedule().withIntervalInMilliseconds(triggerInterval.toMillis).repeatForever().withMisfireHandlingInstructionNowWithRemainingCount())
    .startNow()
    .build()

  def resume(): Unit = {
    scheduler.resumeJob(job.getKey)
    running = true
    logger.debug(s"Started $serviceName service")
  }

  def pause(): Unit = {
    scheduler.pauseJob(job.getKey)
    running = false
    logger.debug(s"Stopped $serviceName service")
  }

  private def schedule(): Unit = {
    logger.debug(s"Scheduled $serviceName service")
    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.rescheduleJob(trigger.getKey, trigger)
    } else {
      scheduler.scheduleJob(job, Set(trigger).asJava, true)
    }
    running = true
  }

  private def unschedule(): Unit = {
    logger.debug(s"Unscheduled $serviceName service")
    if (scheduler.checkExists(trigger.getKey)) {
      scheduler.unscheduleJob(trigger.getKey)
    }
  }

  @event.EventListener
  def onStartup(event: SchedulerStartedEvent): Unit = {
    try {
      if (notificationEnabled) {
        schedule()
      } else {
        unschedule()
      }
    } catch {
      case e: Exception => logger.error(s"Error while starting $serviceName service", e)
    }
  }

  override def enable(): Unit = resume()

  override def disable(): Unit = pause()

  override def isEnabled: Boolean = running

  override def name(): String = serviceName
}
