package com.xebialabs.xlrelease.pendo

import com.xebialabs.analytics.pendo.PendoDataPublisher.{DisablePublishing, EnablePublishing}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.domain.events.{ConfigurationCreatedEvent, ConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.pendo.actor.PendoDataPublisherInitializer
import com.xebialabs.xlrelease.service.{BroadcastService, ConfigurationService}
import grizzled.slf4j.Logging

import java.util.concurrent.atomic.AtomicBoolean
import scala.util.{Failure, Success, Try}

@EventListener
class PendoSwitch(configurationService: ConfigurationService, broadcastService: BroadcastService)
  extends Logging {

  val PendoAnalyticsType = "xlrelease.PendoAnalytics"
  val pendoEnabled: AtomicBoolean = new AtomicBoolean(getPendoEnabled)
  val PENDO_TYPE: Type = Type.valueOf(PendoAnalyticsType)

  def isEnabled: Boolean = pendoEnabled.get()

  @AsyncSubscribe
  def onConfigurationUpdatedEvent(event: ConfigurationUpdatedEvent): Unit = handleSettings(event.updated)

  @AsyncSubscribe
  def onConfigurationCreatedEvent(event: ConfigurationCreatedEvent): Unit = handleSettings(event.conf)

  @AsyncSubscribe
  def onPendoToggleEvent(event: PendoToggleEvent): Unit = togglePendoInternal(event.pendoEnabled)

  private def handleSettings(settings: BaseConfiguration): Unit =
    if (PENDO_TYPE == settings.getType) {
      val enabled: Boolean = settings.getProperty("enabled")
      togglePendo(enabled)
    }

  private def togglePendo(enabled: Boolean): Unit = {
    broadcastService.broadcast(PendoToggleEvent(enabled), publishEventOnSelf = true)
  }

  private def togglePendoInternal(enabled: Boolean): Unit = {
    val msg = if (enabled) EnablePublishing else DisablePublishing
    pendoEnabled.set(enabled)
    PendoDataPublisherInitializer.getInstance() ! msg
    logger.debug(s"Pendo analytics ${if (enabled) "enabled" else "disabled"}")
  }

  private def getPendoEnabled(): Boolean = {
    Try(configurationService.getFeatureSettings(PendoAnalyticsType)) match {
      case Success(pendoSettings) => pendoSettings.getProperty("enabled")
      case Failure(_) => {
        val descriptor = Type.valueOf(PendoAnalyticsType).getDescriptor
        descriptor.getPropertyDescriptor("enabled").getDefaultValue.asInstanceOf[Boolean]
      }
    }
  }
}

case class PendoToggleEvent(pendoEnabled: Boolean) extends DistributedXLReleaseEvent
