package com.xebialabs.xlrelease.pendo.extractors

import com.xebialabs.analytics.pendo.PendoDataExtractor
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.pendo.PendoConstants._
import com.xebialabs.xlrelease.pendo.utils.PendoUtils.{PendoMap, splitPropertyList}
import com.xebialabs.xlrelease.repository.Ids.getName
import com.xebialabs.xlrelease.risk.domain.Risk
import grizzled.slf4j.Logging

import java.util.Date
import java.util.concurrent.TimeUnit
import scala.jdk.CollectionConverters.{CollectionHasAsScala, MapHasAsScala}

trait ReleaseEventExtractor[T] extends PendoDataExtractor[T] with Logging {

  def eventProperties(release: Release): Map[String, Any] = {
    Map(
      RELEASE_ID -> getName(release.getId),
      TIMESTAMP -> new Date
    ).toPendo
  }

}

trait ReleaseFullEventExtractor[T] extends ReleaseEventExtractor[T] {

  override def eventProperties(release: Release): Map[String, Any] = {
    super.eventProperties(release) ++ releaseProperties(release).toPendo
  }

  def durationBetween(date1: Date, date2: Date) = {
    (date1, date2) match {
      case (d1, d2) if d1 != null && d2 != null =>
        val diffInMillies = Math.abs(d1.getTime - d2.getTime)
        Some(TimeUnit.SECONDS.convert(diffInMillies, TimeUnit.MILLISECONDS))
      case _ => None
    }
  }

  def releaseProperties(release: Release): Map[String, Any] = {
    val tasks = release.getAllTasks.asScala.toList
    val allComments = tasks.foldLeft(List[Comment]())((accumulator, t) => accumulator ++ t.getComments.asScala.toList)

    val properties = scala.collection.mutable.Map[String, Any]()
    properties += (START_DATE -> release.getStartDate)
    properties += (END_DATE -> release.getEndDate)
    val plannedDuration = if (null != release.getPlannedDuration) release.getPlannedDuration else
      durationBetween(release.getDueDate, release.getScheduledStartDate).getOrElse(0)
    properties += (PLANNED_DURATION -> plannedDuration)
    val actualDuration = if (null != release.getActualDuration) release.getActualDuration.getStandardSeconds else 0
    properties += (ACTUAL_DURATION -> actualDuration)
    properties += (RISK_SCORE -> getRiskScore(release.getExtensions.asScala.toList))
    properties += (TASKS -> tasks.size)
    properties += (PHASES -> release.getPhases.size)
    properties += (VARIABLES -> release.getVariables.size)
    properties += (VARIABLES_SIZE -> release.getVariables.asScala.filter(variable => !variable.isValueEmpty).foldLeft(0)
    ((totalVarSize, variable) => totalVarSize + calculateVariableSize(variable)).toFloat / 1024)
    properties += (ATTRIBUTES -> tasks.map(_.getFacets.size()).sum)
    properties += (EXTENSIONS -> release.getExtensions.size())
    properties += (COMMENTS -> allComments.size)
    properties += (COMMENTS_SIZE -> allComments.filter(c => c.getText != null && c.getText.nonEmpty)
      .map(_.getText.getBytes("UTF-16").length).sum.toFloat / 1024)
    properties += (ATTACHMENTS -> release.getAttachments.size())
    properties += (DEPENDENCIES -> tasks.filter(t => t.isGate).foldLeft(0)((accum, task) => accum +
      task.asInstanceOf[GateTask].getDependencies.size()))
    properties += (TASK_FAILURES -> tasks.map(_.getFailuresCount).sum)
    properties += (PHASE_RESTARTS -> release.getAutomatedResumeCount)
    val automatedTasks = tasks.count(t => t.isAutomated)
    properties += (AUTOMATED_TASKS -> automatedTasks)
    val automatedPercentage = safeDivision(automatedTasks, tasks.size)
    properties += (AUTOMATION_PERCENTAGE -> automatedPercentage)

    properties.toMap ++ createTaskTypeProperty(taskCountsByType(tasks))
  }

  private def getRiskScore(extensions: List[ReleaseExtension]): Int = {
    extensions.find(entry => entry.isInstanceOf[Risk]).map(_.asInstanceOf[Risk].getTotalScore.toInt).getOrElse(0)
  }

  def taskCountsByType(tasks: List[Task]): List[String] = {
    // TODO ted: check ordering actually works
    val taskGroups = tasks.groupBy(_.getType).view.mapValues(_.size).toSeq.sortBy(_._2)(Ordering.Int.reverse)
    taskGroups.foldLeft(List[String]())((accum, tsk) => accum :+ s"${tsk._1}-${tsk._2}")
  }

  // TODO ted: check if sizes are matching, proper value provider calculation
  private def calculateVariableSize(variable: Object): Integer = {
    variable match {
      case _: BooleanVariable => PRIMITIVE_SIZE
      case _: DateVariable => PRIMITIVE_LONG_DOUBLE_SIZE
      case _: IntegerVariable => PRIMITIVE_SIZE
      case los: ListOfStringValueProviderConfiguration => los.getValues.asScala.foldLeft(0)((total, value) => total + value.length * PRIMITIVE_SIZE)
      case ls: ListStringVariable => ls.getValue.asScala.foldLeft(0)((total, value) => total + value.length * PRIMITIVE_SIZE)
      case mss: MapStringStringVariable => mss.getValue.asScala.foldLeft(0)((total, value) => total + value._1.length * PRIMITIVE_SIZE +
        value._2.length * PRIMITIVE_SIZE)
      case ps: PasswordStringVariable => ps.getValue.length * PRIMITIVE_SIZE
      case ss: SetStringVariable => ss.getValue.asScala.foldLeft(0)((total, value) => total + value.length * PRIMITIVE_SIZE)
      case s: StringVariable => s.getValue.length * PRIMITIVE_SIZE
      case _ => {
        logger.warn(s"Unrecognized variable type ${variable.getClass.getCanonicalName}, not calculating size")
        0
      }
    }
  }

  def createTaskTypeProperty(tasks: List[String]): Map[String, Any] = {
    val propertyArray = splitPropertyList(tasks)
    Map(TASK_TYPES_1 -> propertyArray(0), TASK_TYPES_2 -> propertyArray(1))
  }

  private def safeDivision(divident: Int, divisor: Int): Float = if (divisor != 0) divident.toFloat / divisor else 0

}
