package com.xebialabs.xlrelease.pendo

import com.xebialabs.analytics.pendo.{PendoDataExtractor, PendoEventQueue, PendoSender}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.api.internal.ReleaseServerResource
import com.xebialabs.xlrelease.api.utils.AccountIdHelper.generateAccountId
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.pendo.actor.PendoDataPublisherInitializer
import com.xebialabs.xlrelease.pendo.metrics.PendoMetricsService
import com.xebialabs.xlrelease.pendo.utils.PendoUtils.generatePendoVistorId
import com.xebialabs.xlrelease.service.{BroadcastService, ConfigurationService}
import com.xebialabs.xlrelease.views.LicenseView
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Value}
import org.springframework.context.annotation.{Bean, Configuration}

import java.net.InetAddress
import scala.jdk.CollectionConverters.ListHasAsScala
import scala.language.postfixOps

@Configuration
class XlrPendoConfiguration extends Logging {

  @Autowired
  var extractors: java.util.List[_ <: PendoDataExtractor[_]] = _

  @Autowired
  var configurationService: ConfigurationService = _

  @Autowired
  var broadcastService: BroadcastService = _

  @Autowired
  var serverConfiguration: ServerConfiguration = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var releaseServerResource: ReleaseServerResource = _

  @Value("${xl.features.analytics.pendoQueueSize}")
  var pendoQueueSize: Int = _

  @Bean
  def pendoSwitch = new PendoSwitch(configurationService, broadcastService)

  @Bean
  def pendoRelevantEventsListener(pendoDataHolder: PendoEventQueue, pendoSwitch: PendoSwitch): PendoRelevantEventsListener = {
    val extractorsMap = extractors.asScala.toList map (e => e.supports() -> e) toMap
    val funnel = new PendoRelevantEventsListener(pendoSwitch, pendoDataHolder, extractorsMap)
    funnel
  }

  @Bean
  def pendoEventQueue(): PendoEventQueue = new PendoEventQueue(pendoQueueSize)

  @Bean
  def pendoSender(asHolder: ActorSystemHolder): PendoSender = {
    implicit val actorSystem = asHolder.actorSystem
    val licenseView = new LicenseView(licenseService.getLicense)

    val accountId = generateAccountId(licenseView.getLicensedTo, serverConfiguration.getServerUrl)
    val host = if (XlrConfig.getInstance.cluster.config.hasPath("name")) {
      XlrConfig.getInstance.cluster.config.getString("name")
    } else {
      InetAddress.getLocalHost.getHostName
    }

    val visitorId = generatePendoVistorId(accountId, host)
    new PendoSender(accountId, visitorId, PendoConstants.TRACKED_EVENT_INTEGRATION_KEY)
  }

  @Bean
  def pendoDataPublisherInitializer(queue: PendoEventQueue, sender: PendoSender, asHolder: ActorSystemHolder) = {
    new PendoDataPublisherInitializer(queue, sender, asHolder)
  }

  @Bean
  def pendoMetricsService(dataHolder: PendoEventQueue, pendoSwitch: PendoSwitch): PendoMetricsService = {
    new PendoMetricsService(dataHolder, pendoSwitch)
  }
}

