package com.xebialabs.xlrelease.notifications.validators

import com.google.common.base.Strings.isNullOrEmpty
import com.xebialabs.deployit.plugin.api.validation.{ExtendedValidationContext, ValidationContext, Validator}
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator
import com.xebialabs.xlrelease.notifications.configuration.{BasicSmtpAuthentication, OAuth2SmtpAuthentication, SmtpServer}

class SmtpServerValidator extends Validator[SmtpServer] {
  override def validate(smtpServer: SmtpServer, context: ValidationContext): Unit = {
    val extendedContext = context.asInstanceOf[ExtendedValidationContext]
    if (isNullOrEmpty(smtpServer.host)) {
      extendedContext.error(smtpServer, "host", "SMTP server host cannot be blank")
    }
    if (smtpServer.port < 0 || smtpServer.port > 0xFFFF) {
      extendedContext.error(smtpServer, "port", "SMTP server port is out of range")
    }
    if (isNullOrEmpty(smtpServer.fromAddress)) {
      extendedContext.error(smtpServer, "fromAddress", "From address is required")
    }
    if (!isNullOrEmpty(smtpServer.fromAddress) && !UserAccountValidator.isValidEmailAddress(smtpServer.fromAddress)) {
      extendedContext.error(smtpServer, "fromAddress", s"${smtpServer.fromAddress} is not a valid email address")
    }
    if (!isNullOrEmpty(smtpServer.testAddress)) {
      val recipientsList = smtpServer.getTestAddress.split(",").toList
      recipientsList.foreach { recipient =>
        if (!UserAccountValidator.isValidEmailAddress(recipient)) {
          extendedContext.error(smtpServer, "testAddress", s"$recipient is not a valid email address")
        }
      }
    }

    smtpServer.authentication match {
      case null => extendedContext.error(smtpServer, "authentication", s"Authentication method is required")
      case basicSmtpAuthentication: BasicSmtpAuthentication =>
        this.validate(basicSmtpAuthentication, extendedContext)
      case oauth2SmtpAuthentication: OAuth2SmtpAuthentication =>
        this.validate(oauth2SmtpAuthentication, extendedContext)
      case _ =>
    }
  }

  private def validate(basicSmtpAuthentication: BasicSmtpAuthentication, extendedContext: ExtendedValidationContext): Unit = {
    if (isNullOrEmpty(basicSmtpAuthentication.username)) {
      extendedContext.error(basicSmtpAuthentication, "username", "Username cannot be blank")
    }
    if (isNullOrEmpty(basicSmtpAuthentication.password)) {
      extendedContext.error(basicSmtpAuthentication, "password", "Password cannot be blank")
    }
  }

  private def validate(oauth2SmtpAuthentication: OAuth2SmtpAuthentication, extendedContext: ExtendedValidationContext): Unit = {
    if (isNullOrEmpty(oauth2SmtpAuthentication.accessTokenUrl)) {
      extendedContext.error(oauth2SmtpAuthentication, "accessTokenUrl", "Access Token URL cannot be blank")
    }
    if (isNullOrEmpty(oauth2SmtpAuthentication.clientId)) {
      extendedContext.error(oauth2SmtpAuthentication, "clientId", "Client ID cannot be blank")
    }
    if (isNullOrEmpty(oauth2SmtpAuthentication.clientSecret)) {
      extendedContext.error(oauth2SmtpAuthentication, "clientSecret", "Client secret cannot be blank")
    }
    if (isNullOrEmpty(oauth2SmtpAuthentication.refreshToken)) {
      extendedContext.error(oauth2SmtpAuthentication, "refreshToken", "Refresh token cannot be blank")
    }
  }
}
