package com.xebialabs.xlrelease.notifications.email

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.notifications.configuration.{BasicSmtpAuthentication, OAuth2SmtpAuthentication, SmtpServer}
import com.xebialabs.xlrelease.service.ConfigurationService
import grizzled.slf4j.Logging
import jakarta.mail.internet.MimeMessage
import jakarta.mail.{Authenticator, PasswordAuthentication, Session}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.nio.charset.StandardCharsets
import java.util.{Optional, Properties}

@Component
class EmailFactory @Autowired()(passwordEncrypter: PasswordEncrypter, configurationService: ConfigurationService,
                                refreshTokenOAuth2Helper: RefreshTokenOAuth2Helper) extends Logging {

  private val SOCKET_TIMEOUT = 30000

  def newMimeEmail(smtpServer: SmtpServer, username: Optional[String], password: Optional[String]): MimeMessage = {
    val overrideCredentials = hasOverriddenCredentials(username, password)

    val props = new Properties()
    props.put("mail.smtp.host", smtpServer.host)
    props.put("mail.smtp.port", smtpServer.port.toString)
    props.put("mail.smtp.starttls.enable", smtpServer.tls.toString)
    props.put("mail.smtp.connectiontimeout", SOCKET_TIMEOUT.toString)
    props.put("mail.smtp.timeout", SOCKET_TIMEOUT.toString)
    props.put("mail.mime.charset", StandardCharsets.UTF_8.name)

    if (!overrideCredentials) {
      refreshTokenOAuth2Helper.refreshTokenIfRequired(smtpServer)
    }

    val authenticator = getAuthentication(smtpServer, username, password)
    if (authenticator != null) {
      props.put("mail.smtp.auth", "true")
      if (overrideCredentials) {
        props.setProperty("mail.smtp.auth.mechanisms", "LOGIN PLAIN DIGEST-MD5 NTLM")
        props.setProperty("mail.smtp.auth.xoauth2.disable", "true")
      } else {
        smtpServer.authentication match {
          case _: OAuth2SmtpAuthentication =>
            props.put("mail.smtp.auth.mechanisms", "XOAUTH2")
          case _ =>
        }
      }
    }

    val session = Session.getInstance(props, authenticator)
    new MimeMessage(session)
  }

  private def getPassword(password: String): String = {
    if (passwordEncrypter.isEncoded(password)) {
      passwordEncrypter.decrypt(password)
    } else {
      password
    }
  }

  private def hasOverriddenCredentials(username: Optional[String], password: Optional[String]): Boolean = {
    username.isPresent && !username.get.isEmpty && password.isPresent && !password.get.isEmpty
  }

  private def getAuthentication(smtpServer: SmtpServer, username: Optional[String], password: Optional[String]): Authenticator = {
    if (hasOverriddenCredentials(username, password)) {
      new Authenticator {
        override protected def getPasswordAuthentication(): PasswordAuthentication = {
          new PasswordAuthentication(username.get, getPassword(password.get))
        }
      }
    } else {
      smtpServer.authentication match {
        case basicSmtpAuthentication: BasicSmtpAuthentication =>
          new Authenticator {
            override protected def getPasswordAuthentication(): PasswordAuthentication = {
              new PasswordAuthentication(basicSmtpAuthentication.username, getPassword(basicSmtpAuthentication.password))
            }
          }
        case oauth2SmtpAuthentication: OAuth2SmtpAuthentication =>
          new Authenticator {
            override protected def getPasswordAuthentication(): PasswordAuthentication = {
              new PasswordAuthentication(smtpServer.getFromAddress, getPassword(oauth2SmtpAuthentication.getAccessToken))
            }
          }
        case _ => null
      }
    }
  }
}
